# coding: utf-8
"""\
Функции, помогающие в разборе конфигурационного файла
"""
import toml
from os.path import exists
from typing import Union, Any

from .error import Error
from .type_helpers import BaseTypeHelper


class ConfigFileError(Error):
    """\
    Базовое исключение при обработке конфигурационного файла
    """


class ConfigSectionNotFound(ConfigFileError):
    """\
    Отсутствие нужного раздела в конфигурации
    """


class ConfigFileContent(object):
    """\
    Работа с содержимым файла конфигурации
    """
    def __init__(self, file_data, section_name=None):
        """\
        :param file_data: Разобранное в словарь содержимое файла конфигурации
        :param section_name: Имя секции в конфигурации
        """
        self.d = file_data
        self.section = section_name

    def _format_err_msg(self, msg: str) -> str:
        if self.section is not None:
            return f'Раздел конфигурации "{self.section}": {msg}'

        else:
            return msg

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        pass

    def get_value(self, name: str,
                  val_type: Union[type, BaseTypeHelper] = str,
                  default: Any = None,
                  mandatory: bool = True):
        """\
        Получить из текущего раздела конфигурации нужный элемент
        :param name: имя параметра
        :param val_type: тип параметра
        :param default: значение параметра по умолчанию
        :param mandatory: является ли параметр обязательным. Если ``False``, то в случае отсутствия
                          параметра будет использовано значение None
        """
        if name not in self.d:
            if mandatory and default is None:
                raise ConfigFileError(self._format_err_msg(f'Параметр "{name}" в файле отсутствует'))

            else:
                return default

        else:
            _buf = self.d[name]
            try:
                return val_type(_buf)

            except (ValueError, TypeError) as e:
                raise ConfigFileError(self._format_err_msg(f'Не удалось привести значение "{_buf}" к типу '
                                                           f'{repr(val_type)}: {e}'))

    def get_section(self, name: str, mandatory: bool = True):
        """\
        Получить раздел конфигурационного файла из текущего или корневого раздела
        :param name: имя раздела
        :param mandatory: является ли раздел обязательным или он может отсутствовать в конфигурации. Во втором
                          случае, будет создан пустой словарь и использован в качестве данных раздела.
        :returns Экземпляр того-же класса, только для новой секции
        """
        if name not in self.d:
            if mandatory:
                raise ConfigSectionNotFound(self._format_err_msg(f'Раздел "{name}" не найден'))

            else:
                return ConfigFileContent(file_data={}, section_name=f'{self.section}.{name}')

        else:
            _val = self.d[name]

            if not isinstance(_val, dict):
                raise ConfigFileError(self._format_err_msg(f'Параметр конфигурации "{name}" не является разделом'))

            else:
                return ConfigFileContent(file_data=_val, section_name=f'{self.section}.{name}')


class ConfigFile(object):
    """\
    Работа с самим файлом конфигурации
    """
    def __init__(self, file_name: str):
        if not exists(file_name):
            raise ConfigFileError(f'Файл конфигурации "{file_name}" не существует')

        self.file_name = file_name

    def get(self):
        """\
        Производим разбор файла конфигурации и создаём корневой раздел конфигурации
        """
        try:
            return ConfigFileContent(toml.load(self.file_name))

        except OSError as e:
            raise ConfigFileError(f'Ошибка чтения файла "{self.file_name}": {e}')

        except toml.TomlDecodeError as e:
            raise ConfigFileError(f'Ошибка в разборе файла конфигурации: {e}')

        except ValueError as e:
            raise ConfigFileError(f'Ошибка разбора файла конфигурации: {ConfigFileError.fmt_error(e)}')

    def __exit__(self, exc_type, exc_val, exc_tb):
        pass

    def __enter__(self):
        return self.get()
