# coding: utf-8
"""\
Инструмент миграции схемы БД.

"""

from os.path import exists, join as p_join, isdir
from os import listdir

from . import Error


class MigrateError(Error):
    """\
    Общий класс ошибок миграции
    """


class MigrateManager(object):
    """\
    Менеджер миграции
    """
    def __init__(self, control_table: str, migrate_env: str):
        """
        :param control_table: Имя таблицы, хранящей метаданные миграции
        :param migrate_env: Директория, хранящая SQL-скрипты миграции
        """
        self.control_table = control_table

        if not exists(migrate_env):
            raise MigrateError('Migrate enviroment not found')

        self.schema = p_join(migrate_env, 'schema.sql')
        if not exists(self.schema):
            raise MigrateError(f'Schema file not found: {self.schema}')

        self.patch_dir = p_join(migrate_env, 'patch')
        if not isdir(self.patch_dir):
            raise MigrateError(f'Patch dir not found or not directory: {self.patch_dir}')

    def get_patch_files(self, ver: int):
        """\
        Получение из директории файлов миграции списка применяемых к данному экземпляру БД
        """
        res = {}
        for f in listdir(self.patch_dir):
            if not f.lower().endswith('.sql'):
                continue

            _f = f.strip().split('.')

            try:
                _ver = int(_f[0])

            except (TypeError, ValueError) as e:
                raise MigrateError(f'Error on parse version "{_f[0]}" of file "{f}": {e}')

            except IndexError:
                raise MigrateError(f'Error on get version from filename: {f}')

            if _ver in res:
                raise MigrateError(f'Version duplicates on parse file: {f}')

            res[_ver] = p_join(self.patch_dir, f)

        for i in sorted(res.keys()):
            if i > ver:
                yield i, res[i]

    @staticmethod
    def get_commands(file: str):
        """\
        Получение из файлов серий команд, которые необходимо применять на БД
        """
        buf = []
        with open(file) as IN:
            for l in IN:
                if l.lstrip().startswith('--'):
                    if buf:
                        yield '\n'.join(buf)
                        buf[:] = []

                else:
                    buf.append(l)

        if buf:
            yield '\n'.join(buf)

    def init_db(self, db):
        """\
        Инициализация БД.

        :param db: Объект-подключения, представляющий нужную БД и при этом поддерживающий DB API Python
        """
        cursor = db.cursor()
        for c in self.get_commands(self.schema):
            cursor.execute(c)
            db.commit()

        db.commit()

    def check(self, db):
        """\
        Проверка БД на соответствие.

        :param db: Объект-подключения, представляющий нужную БД и при этом поддерживающий DB API Python
        """
        cursor = db.cursor()
        cursor.execute(f"SELECT version FROM {self.control_table}")
        q = cursor.fetchone()
        del cursor

        if q is None:
            ver = -1
        else:
            ver = int(q[0])

        new_ver = ver
        cursor = db.cursor()
        for up_ver, patch_file in self.get_patch_files(ver):
            new_ver = up_ver
            for cmd in self.get_commands(patch_file):
                cursor.execute(cmd)
                db.commit()

        cursor.execute(f"DELETE FROM {self.control_table}")

        cursor.execute(f"""
            INSERT INTO {self.control_table} (version)
            VALUES ({new_ver})
        """)
        db.commit()

    @staticmethod
    def get_conn_from_my_obj(obj: object):
        """\
        Получиение объекта соединения из обёрток, которые я сам себе пишу для работы с DB-API

        :param obj: Получение объекта-подключения из объектов БД своего стиля оформления.
        :return:
        """
        if hasattr(obj, '_conn'):
            return obj._conn
        elif hasattr(obj, '_con'):
            return obj._con
        else:
            raise TypeError('No known connection object in given database object found')
