# coding: utf-8
# devel.a0fs.ru -- aw_log -- v0.r202402.1
""" Реализация классов логирования

Метки в журнале о уровне сообщения:
  "`": Debug
  ".": Info
  "*": Warning
  "!": Error
  "#": Alert

"""

from time import monotonic
from datetime import timedelta
from traceback import extract_tb, extract_stack
from typing import Optional, Any, Iterable
from sys import exc_info


class Timing(object):
    def __init__(self, name: Optional[str] = None):
        if name is None:
            self.prefix = ''

        else:
            self.prefix = f'{name} :: '

        self.tsAll = monotonic()
        self.ts = self.tsAll

    def get_time(self):
        return monotonic() - self.ts

    def reset(self):
        self.ts = monotonic()
        self.tsAll = self.ts

    def __str__(self):
        ts = monotonic()
        return self.prefix + '%s(%.4f)' % (timedelta(seconds=(ts - self.tsAll)), ts - self.ts)

    def __call__(self, msg):
        _buf = f'{self} | {msg}'
        self.ts = monotonic()
        return _buf


class AbstractLogBase(object):
    def __init__(self, prefix: str = 'main'):
        self.prefix = prefix

    def _write_helper(self, mark: str, msg: Any) -> Iterable[str]:
        for l in str(msg).splitlines():
            yield f'{mark} {self.prefix} | {l}\n'

    def _write(self, mark: str, msg: Any):
        raise NotImplemented(f'Метод write не определён для класса "{type(self).__name__}"')

    def __call__(self, msg):
        self._write('.', msg)

    def err(self, msg):
        self._write('!', msg)

    def warn(self, msg):
        self._write('*', msg)

    def alert(self, msg):
        self._write('#', msg)

    def debug(self, msg):
        self._write('`', msg)

    @staticmethod
    def get_timing(name: Optional[str] = None):
        return Timing(name)

    def sub_log(self, name: str):
        return self.__class__(f'{self.prefix}/{name}')

    def excpt(self, msg, e_class=None, e_obj=None, e_tb=None, stack_skip=0):
        if e_class is None:
            e_class, e_obj, e_tb = exc_info()

        tb_data_tb = list(extract_tb(e_tb))[::-1]
        tb_data_stack = list(extract_stack())[::-1][(2 + stack_skip):]
        self.err(msg)
        self.err('--- EXCEPTION ---')
        self.err(f' {e_class.__name__} ({e_obj})')
        self.err('--- TRACEBACK ---')
        for _tb_file, _tb_line, _tb_func, _tb_text in tb_data_tb:
            self.err(f'File: {_tb_file}, line {_tb_line} in {_tb_func}')
            self.err(f'   {_tb_text}')

        self.err('>>> Exception Handler <<<')
        for _tb_file, _tb_line, _tb_func, _tb_text in tb_data_stack:
            self.err(f'File: {_tb_file}, line {_tb_line} in {_tb_func}')
            self.err(f'   {_tb_text}')

        self.err('--- END EXCEPTION ---')
