# coding: utf-8

import bottle
from typing import Union, Any, Dict, Optional
from datetime import datetime


class Cookie(object):
    """\
    Класс хранящий ``cookie`` и способный их устанавливать в объекты http-ответов ``bottle``
    """

    def __init__(self, name: str, value: str,
                 max_age: Optional[int] = None,
                 expires: Optional[Union[int, datetime]] = None,
                 path: Optional[str] = None,
                 secure: bool = True,
                 httponly: bool = True,
                 samesite: bool = False,
                 domain: Optional[str] = None):
        """\
        :param name: Имя ``cookie``
        :param value: Значение ``cookie``
        :param max_age: Время жизни ``cookie`` в секундах.
        :param expires: Значение времени, когда cookie будет удалена, задаётся в виде ``unix timestamp (int)`` или
                        ``datetime``
        :param path: Префикс пути поиска ресурса на данном сайте, для которого следует отправлять данное ``cookie``
        :param secure: Отправлять ``cookie`` только по шифрованным каналам связи
        :param httponly: Сделать ``cookie`` не доступной для ``JavaScript``
        :param samesite: Не отправлять данную cookie, если запрос пришёл не с того же сайта (анализируется заголовок
                         referer)
        :param domain: Имя домена в рамках которого выставляется cookie. В современных браузерах может и глючить
        """

        self.name = name
        self.value = value
        self.max_age = max_age
        self.expires = expires
        self.path = path
        self.secure = secure
        self.httponly = httponly
        self.samesite = samesite
        self.domain = domain

    def to_dict(self) -> Dict[str, Any]:
        """\
        Подготавливает параметры, которые можно передать в процедуру ``set_cookie``
        объекта ``Response``
        """
        res = {
            'name': self.name,
            'value': self.value,
            'secure': self.secure,
            'httponly': self.httponly,
        }

        if self.samesite:
            res['samesite'] = 'strict'

        for k in ('max_age', 'expires', 'path', 'domain'):
            if getattr(self, k) is not None:
                res[k] = getattr(self, k)

        return res

    def get_remove_param(self) -> Dict[str, Any]:
        """\
        Подготавливает параметры, которые можно передать в процедуру ``delete_cookie``
        объекта ``Response``
        """
        res = self.to_dict()
        res['key'] = res['name']
        del res['value']
        del res['name']
        return res

    def response_add(self, resp: bottle.BaseResponse):
        resp.set_cookie(**self.to_dict())

    def response_delete(self, resp: bottle.BaseResponse):
        resp.delete_cookie(**self.get_remove_param())
