# coding: utf-8
"""\
Поддержка работы с системой авторизации [authelia](https://www.authelia.com/)
"""
import bottle
from typing import Optional, List
from collections import namedtuple

from . import Error


# Константы
HEADER_USER = 'Remote-User'      # Заголовок по умолчанию, в котором система аутентификации передаёт имя УЗ пользователя
HEADER_GROUPS = 'Remote-Groups'  # Заголовок по умолчанию, в котором система аутентификации передаёт группы пользовтеля
HEADER_EMAIL = 'Remote-Email'    # Заголовок по умолчанию, в котором система аутентификации передаёт e-mail пользователя
HEADER_NAME = 'Remote-Name'      # Заголовок по умолчанию, в котором система аутентификации передаёт имя пользователя


class AHError(Error):
    """\
    Базовый класс исключений модуля адаптера Authelia
    """


class AHAuthError(AHError):
    """\
    Проблемы с авторизацией пользователя
    """
    def __init__(self):
        super().__init__('Пользователь не авторизован')


AHUser = namedtuple('AHUser', ['uname', 'groups', 'email', 'name'])


class AutheliaHelper(object):
    def __init__(self,
                 group_filter: Optional[List[str]] = None,
                 header_user: str = HEADER_USER,
                 header_groups: str = HEADER_GROUPS,
                 header_email: str = HEADER_EMAIL,
                 header_name: str = HEADER_NAME
                 ):
        """\
        :param group_filter: Фильтр передаваемый системой аутентификации групп, которые интересны нам. Ели ``None``
                             группы, передаваемые в заголовке не фильтруются
        :param header_user: Имя заголовка с именем УЗ пользователя.
        :param header_groups: Имя заголовка с группами пользователя
        :param header_name: Имя заголовка с именем пользователя
        """
        self.header_user = header_user
        self.header_groups = header_groups
        self.header_email = header_email
        self.header_name = header_name

        self.group_filter = None
        self.group_filter_set = None
        if group_filter is not None:
            self.group_filter = {}
            for i in group_filter:
                self.group_filter[i.lower()] = i

            self.group_filter_set = set(self.group_filter.keys())

    def __call__(self, request: bottle.BaseRequest = bottle.request) -> AHUser:
        """\
        Обработка запроса. Если заголовки системы авторизации присутствуют, информацией из них заполняется
        объект ``AHUser``. Если данные не найдены, то будет возбуждено исключение.

        :param request: Обрабатываемый запрос
        :returns: Объект ``AHUser`` с данным авторизации.
        """

        uname = request.get_header(self.header_user)
        email = request.get_header(self.header_email)
        name = request.get_header(self.header_name)
        groups = request.get_header(self.header_groups)

        if uname is None:
            raise AHAuthError()

        if groups is not None:
            groups = tuple(map(lambda x: x.stip(), groups.split(',')))
            if self.group_filter is not None:
                _groups = []
                _buf = set(map(lambda x: x.lower(), groups))

                for grp in _buf & self.group_filter_set:
                    try:
                        _groups.append(self.group_filter[grp])

                    except KeyError:
                        AHError(f'Ошибка в фильтре групп. Для ключа "{grp}" не нашлось значения, хотя оно присутствует '
                                f'в множестве возможных значений.')

        return AHUser(uname=uname, groups=groups, name=name, email=email)
