# coding: utf-8
"""\
Модуль вспомогательных средств обеспечения идентификации пользователей
"""
import bottle
from string import ascii_letters, digits
from typing import Optional, Dict, Iterable, Any
from . import btle_tools as tools
from . import Error
from . import jwt_helper
from .cookie import Cookie


SESSION_TIMEOUT = 86400    # Продолжительность сессии по умолчанию.


class IDError(Error):
    """\
    Общий класс ошибок идентификации
    """


class IDCheckError(IDError):
    """\
    Ошибка проверки ID
    """


class IDNotFound(IDCheckError):
    """\
    Запрос не идентифицирован
    """
    def __init__(self):
        super().__init__('Запрос не идентифицирован')


class UIDError(IDError):
    """\
    Ошибки при работе с классом пользовательского идентификатора
    """


class UID(object):
    """\
    Класс пользовательского идентификатора.

    Обеспечивает хранение идентификационной информации о пользователе.
    """
    ENV_NAME = 'AW_UID'  # Идентификатор для хранения объекта в контексте запроса.

    def __init__(self, uname: str, acc_level: int = -1,
                 sess_id: Optional[str] = None,
                 acc_tags: Optional[Iterable[str]] = None,
                 user_meta: Optional[Dict[str, str]] = None
                 ):
        """\
        :param uname: Имя пользователя внутри системы
        :param acc_level: Уровень допуска пользователя. Интерпретация целиком на стороне приложения.
        :param sess_id: Идентификатор сессии пользователя
        :param acc_tags: Строковые константы, определяющие конкретное множество доступных пользователю ресурсов.
                        Интерпретация целиком на стороне приложения
        :param user_meta: Иная полезная приложению информация, сохранённая в виде тег-значение
        """
        self.name = uname
        self.level = acc_level
        self.sess_id = sess_id
        self._access = acc_tags
        self._meta = user_meta

    def get_fp_id(self) -> str:
        """\
        Преобразует объект в форму, пригодную для получения подписи сессии
        """
        buf = ':'.join(sorted(map(lambda x: str(x).lower().strip(), self.get_access())))
        if self.level >= 0:
            buf += f':{self.level}'

        return f'{self.name}:{buf}'

    def __str__(self):
        return self.name

    def get_access(self):
        """\
        Получить теги доступа пользователя, если они имелись
        """
        if self._access:
            return [i for i in self._access]

        else:
            return []

    def get_meta(self):
        """\
        Получить метаданные пользователя, если они имелись.
        """
        if self._meta:
            return dict([(k, v) for k, v in self._meta.items()])

        else:
            return dict()

    def to_dict(self) -> Dict[str, str]:
        """\
        Преобразование текущего состояния объекта в массив
        """
        res = dict(
            n=self.name,
            id=self.sess_id
        )

        if self.level >= 0:
            res['al'] = self.level

        if self._access:
            res['a'] = self.get_access()

        if self._meta:
            res['m'] = self.get_meta()

        return res

    def to_env(self, request: bottle.BaseRequest = bottle.request):
        """\
        Сохранение своего экземпляра в контекст запроса
        """
        tools.set_env(self.ENV_NAME, self, request=request)

    @classmethod
    def from_dict(cls, d: Dict[str, Any]):
        """\
        Разбор словаря, созданного методом ``to_dict`` и восстановление из него экземпляра объекта,
        подобного сохранённому
        """
        params = dict()

        # User name
        uname = d.get('n')
        try:
            uname = str(uname) if uname is not None else ''

        except (ValueError, TypeError) as e:
            raise UIDError(f'Не удалось представить имя пользователя как строку: name="{uname}" err="{e}"')

        if not uname:
            raise UIDError(f'Идентификатор пользователя не задан в переданном словаре')

        params['uname'] = uname

        # Session Identity
        sess_id = d.get('id')
        if sess_id is not None:
            try:
                sess_id = str(sess_id)

            except (TypeError, ValueError) as e:
                UIDError(f'Не удалось представить идентификатор сессии как строку: '
                         f' sess_id="{sess_id}" '
                         f' err="{e}"')

        params['sess_id'] = sess_id

        # Access Level
        level = d.get('al')

        if level is not None:
            if type(level) is not int:
                try:
                    level = int(level)

                except (TypeError, ValueError) as e:
                    raise UIDError(f'Не получилось представить уровень допуска пользователя в качестве числа:'
                                   f' level="{level}" err="{e}"'
                                   )

            params['acc_level'] = level

        # Access Tags
        acc_tags = d.get('a')

        if acc_tags is not None:
            try:
                acc_tags = (t for t in iter(acc_tags))

            except (ValueError, TypeError) as e:
                raise UIDError(f'Не вышло преобразовать теги доступа к строковому значению: '
                               f' acc_tags="{acc_tags}" '
                               f' err="{e}"')

            params['acc_tags'] = acc_tags

        # User Metadata
        u_meta = d.get('m')

        if u_meta is not None:
            if type(u_meta) is not dict:
                raise UIDError(f'Пользовательские метаданные не имеют нужного типа: {type(u_meta).__name__}')

            try:
                u_meta = dict((str(k), str(v)) for k, v in u_meta.items())

            except (TypeError, ValueError) as e:
                raise UIDError(f'Не удалось преобразовать в словарь пользовательские метаданные: {e}')

            params['user_meta'] = u_meta

        return cls(**params)

    @classmethod
    def from_env(cls, request: bottle.BaseRequest = bottle.request):
        """\
        Извлечение экземпляра объекта из контекста запроса, если он там сохранён.
        """
        uid = tools.get_env(cls.ENV_NAME, request=request)

        if uid is None:
            raise IDNotFound()

        if type(uid) is not cls:
            raise IDCheckError(f'Неожиданный тип ID: type="{type(uid).__name__}" val="{uid}"')

        return uid


class IDHelper(object):
    """\
    Класс поддержки идентификации
    """
    def __init__(self,
                 app_name: str,
                 sign_secret: str,
                 sess_timeout: int = SESSION_TIMEOUT
                 ):
        """\
        :param app_name: Имя приложения. Допускаются цифры символы латиницы и знак ``-``
        :param sign_secret: Секрет, которым будет подписываться JWT.
        :param sess_timeout: Время жизни сессии пользователя.
        """
        if set(ascii_letters + digits + '-') < set(app_name):
            _buf = ', '.join(map(lambda x: f'"{x}"', set(app_name) - set(ascii_letters + digits + '-')))
            raise IDError(f'Не допустимые символы в имени приложения: {_buf}')

        self.app_name = app_name
        self.cookie_name = f'X-ID-{self.app_name}'
        self.sess_timeout = sess_timeout
        self.jwt_helper = jwt.JWTHelper(sign_secret)

        # Свойства необходимые для работы в качестве плагина Bottle
        self.name = 'IDHelper'
        self.api = 2

    def make_cookie(self, uid: Optional[UID] = None,
                    request: bottle.BaseRequest = bottle.request,
                    is_secure: bool = True,
                    is_httponly: bool = True
                    ) -> Cookie:
        """\
        Формируем ``Cookie`` для идентификации сессии пользователя.

        :param uid: Экземпляр ``UID``, идентифицирующий сессию. Если нет получаем из контекста запроса
        :param request: Экземпляр запроса, на основании которого формируется ответ.
        :param is_secure: Установить признак ``secure`` на cookie
        :param is_httponly: Установить признак ``httponly`` на cookie
        """
        if uid is None:
            uid = UID.from_env(request=request)

        uid_dict = uid.to_dict()
        uid_dict['fp'] = tools.get_session_fingerprint(uid.get_fp_id(), request=request)

        return Cookie(
            name=self.cookie_name,
            value=self.jwt_helper.encode(uid.to_dict(), timeout=self.sess_timeout),
            max_age=self.sess_timeout,
            secure=is_secure,
            httponly=is_httponly
        )

    def set_id(self, uid: Optional[UID] = None, response: bottle.BaseResponse = bottle.response,
               request: bottle.BaseRequest = bottle.request,
               is_secure: bool = True,
               is_httponly: bool = True
               ):
        """\
        Установка cookie на ответ пользователю

        :param uid: Объект ``UID``, которым идентифицируется сессия. По умолчанию берётся из контекста запроса.
        :param response: Объект ответа, на который устанавливаем cookie. По умолчанию ``bottle.response``
        :param request: Экземпляр запроса, на основании которого формируется ответ. По умолчанию ``bottle.request``
        :param is_secure: Установить признак ``secure`` на cookie
        :param is_httponly: Установить признак ``httponly`` на cookie
        """
        cookie = self.make_cookie(uid=uid, request=request, is_secure=is_secure, is_httponly=is_httponly)
        cookie.response_add(response)

    def _get_id_impl(self, request: bottle.BaseRequest = bottle.request):
        """\
        Реализация получения ID и сохранения его в контекст запроса для последующего применения
        в других методах класса.
        """
        sid = tools.get_cookie(self.cookie_name)
        if sid is None:
            raise IDNotFound()

        try:
            uid_raw = self.jwt_helper.decode(sid, check_timeout=True)
            uid = UID.from_dict(uid_raw)

            fp = tools.get_session_fingerprint(uid.get_fp_id(), request=request)

            if fp != uid_raw.get('fp'):
                raise IDCheckError('Проверка подписи сессии не прошла')

            uid.to_env(request=request)

        except (jwt.JWTAuthError, UIDError) as e:
            raise IDCheckError(f'Ошибка проверки ID: {e}')

        except (jwt.JWTError, IDError) as e:
            raise IDError(f'Ошибка в обработке ID запроса: {e}')

    def need_id(self):
        """\
        Декоратор для конкретных точек входа приложения
        """
        def d(callback):
            def f(*a, **kwa):
                self._get_id_impl()
                return callback(*a, **kwa)

            return f

        return d

    @staticmethod
    def env_id(request: bottle.BaseRequest = bottle.request) -> UID:
        """\
        Получаем ID из контекста запроса (для случая, когда проверка и валидация ID проведена на уровне
        плагина или декоратора
        """
        return UID.from_env(request=request)

    # Реализация методов для работы в качестве Bottle плагина
    def setup(self, app: bottle.Bottle):
        pass

    def apply(self, callback: Any, route: bottle.Route):
        def f(*a, **kwa):
            self._get_id_impl()

            return callback(*a, **kwa)

        return f
