# coding: utf-8

import jwt
from datetime import datetime, timedelta
from typing import Optional

from . import Error

JWT_HASH_ALGO = 'HS512'


class JWTError(Error):
    pass


class JWTAuthError(JWTError):
    """\
    Провалена проверка токена на допустимость по подписи, времени действия или прочее
    """


class JWTHelper(object):
    def __init__(self, key: str):
        self.key = key

    def encode(self, data: dict, timeout: Optional[int] = None) -> str:
        if timeout is not None:
            data['exp'] = datetime.utcnow() + timedelta(seconds=timeout)

        return jwt.encode(data, key=self.key, algorithm=JWT_HASH_ALGO)

    def decode(self, token: str, check_timeout: bool = False) -> dict:
        opts = {
            'algorithms': [JWT_HASH_ALGO, ]
        }

        if check_timeout:
            opts['options'] = {'require': {'exp'}}

        if token is None:
            raise JWTAuthError('Ключ отсутствует')
        else:
            token = token.encode('utf-8')

        try:
            return jwt.decode(jwt=token, key=self.key, **opts)

        except (jwt.InvalidIssuerError, jwt.InvalidSignatureError, jwt.ExpiredSignatureError) as e:
            raise JWTAuthError(str(e))

        except jwt.PyJWTError as e:
            raise JWTError(f'{type(e).__name__}: {e}')

    @classmethod
    def make_fabric(cls, key: str):
        def f():
            return cls(key=key)

        return f
