# coding: utf-8
"""\
Инструментарий, призванный помочь в кодировании генерации страниц, посредством
``Bottle.SimpleTemplate``
"""

from bottle import SimpleTemplate, TemplateError
from os.path import join as p_join, isdir
from os import getcwd
from threading import RLock

_SPINLOCK = RLock()
TMPL_ROOT = (getcwd(),)


class SimpleTmplWrapper(object):
    def __init__(self, file_name: str, **static_params):
        """
        :param file_name: Имя файла шаблона, по которому его сможет отыскать класс ``SimpleTemplate``
        :param static_params: Параметры, добавляемые в шаблон страницы при каждой генерации
        """
        try:
            self.tmpl = SimpleTemplate(name=file_name, lookup=TMPL_ROOT)

        except TemplateError as e:
            raise ValueError(f'Ошибка в инициализации шаблона: {e}')

        self.static_params = static_params

    def __call__(self, **params):
        """\
        Сгенерировать страницу по шаблону, обёткой которого является этот класс

        :param params: Параметры шаблона, которые обновляют заданные при инициализации класса параметры и передаются
                    в метод ``SimpleTemplate.render()`` через параметры ключ-значение.
        :return: Результат ``SimpleTemplate.render()`` - сгенерированную страницу
        """
        kw_params = self.static_params.copy()
        kw_params.update(params)
        return self.tmpl.render(**kw_params)

    @staticmethod
    def set_root(*a):
        """\
        Устанавливаем новый корневой каталог, относительного которого будет происходить поиск имён шаблонов
        при их инициализации.

        :param *a: Компоненты пути (имена каталогов, его сотавляющих) или имя директории.
        """
        global TMPL_ROOT

        new_root = p_join(*a)
        if not isdir(new_root):
            raise ValueError(f'Директория, устанавлиеемая как новый корневой каталог, не существует: {new_root}')

        with _SPINLOCK:
            TMPL_ROOT = (new_root,)

    @staticmethod
    def add_root(*a):
        """\
        Аналогичнен методу ``set_root()``, за тем исключением, что путь добавляется к имеющемуся корню,
        а не заменяет его.

        :param *a: Компоненты пути (имена каталогов, его сотавляющих) или имя директории.
        """
        global TMPL_ROOT

        new_root = p_join(*a)
        if not isdir(new_root):
            raise ValueError(f'Директория, устанавлиеемая как новый корневой каталог, не существует: {new_root}')

        with _SPINLOCK:
            TMPL_ROOT += (new_root,)
