# coding: utf-8
"""
Смена пароля на пользователя в AD посредством Python

"""

from ldap3 import Server, Connection, SIMPLE, SUBTREE
from ldap3.core.exceptions import LDAPException
from secrets import token_urlsafe
import string


class LdapError(Exception):
    @staticmethod
    def err_fmt(e):
        _buf = ' '.join(str(e).split())  # всё в одну строку с одним пробелом
        # между словами
        return "%s(%s)" % (type(e).__name__, _buf)


class LdapServerError(LdapError):
    pass


class LdapPassGenError(LdapError):
    pass


class LdapNotFound(LdapError):
    pass


class LdapPasswdChanger(object):
    def __init__(self, ldap_server_uri, ldap_user, ldap_passwd, ldap_base_dn, ldap_timeout=120, ldap_reconnect=5,
                 passwd_len=25):
        self.ldap_server_uri = ldap_server_uri  # URI к серверу: ldaps://server_name:port_number
        self.ldap_user = ldap_user  # Пользователь, имеющий право сброса паролей
        self.ldap_passwd = ldap_passwd  # Пароль к тому пользователю
        self.ldap_base_dn = ldap_base_dn
        self.ldap_timeout = ldap_timeout  # Таймаут операций с сервером
        self.passwd_len = passwd_len  # Желаемая длинна пароля
        self.ldap_reconnect = ldap_reconnect  # Количество попыток переподключения к серверу

        # Заготовка под пароли
        self._set_punct = set(string.punctuation)
        self._set_uper = set(string.ascii_uppercase)
        self._set_lower = set(string.ascii_lowercase)

    def get_passwd(self):
        _cnt = 0
        while True:
            res = token_urlsafe(self.passwd_len)
            _res_set = set(res)
            if (
                    (_res_set & self._set_punct)
                    and (_res_set & self._set_uper)
                    and (_res_set & self._set_lower)
            ):
                return res

            _cnt += 1
            if _cnt > 60:
                raise LdapPassGenError('Не удалось сгенерировать пароль')

    def change_passwd(self, login):
        _reconnects = self.ldap_reconnect
        if _reconnects is None or _reconnects < 1:
            _reconnects = 1

        while True:
            _reconnects -= 1
            try:
                ldap_srv = Server(self.ldap_server_uri, connect_timeout=self.ldap_timeout)
                ldap_conn = Connection(ldap_srv, authentication=SIMPLE,
                                       user=self.ldap_user, password=self.ldap_passwd,
                                       check_names=True,
                                       auto_referrals=False, raise_exceptions=True, auto_range=True,
                                       )
                ldap_conn.open()
                if not ldap_conn.bind():
                    continue

                break

            except LDAPException as e:
                if not _reconnects > 0:
                    raise LdapServerError(f'Ошибка подключения к серверу: {LdapServerError.err_fmt(e)}')

        filter = f'(sAMAccountName={login})'
        try:
            if not ldap_conn.search(self.ldap_base_dn, filter):
                raise LdapServerError('Сервер вернул статус ошибки: %(desc)s(%(code)s) - %(msg)s' % {
                    'desc': ldap_conn.result['description'],
                    'code': ldap_conn.result['result'],
                    'msg': ldap_conn.result['message'],
                })

        except LDAPException as e:
            raise LdapServerError(f'Ошибка при поиска УЗ в ldap: {LdapServerError.err_fmt(e)}')

        if not len(ldap_conn.entries) > 0:
            raise LdapNotFound(f'Не найдена учётная запись: {login}')

        elif len(ldap_conn.entries) != 1:
            _found_names = '; '.join([i.entry_dn for i in ldap_conn.entries[:10]])

            raise LdapError(f'Найдено более одной УЗ по заданному имени {login}: {len(ldap_conn.entries)} '
                            f'- {_found_names}')

        login_dn = ldap_conn.entries[0].entry_dn
        new_passwd = self.get_passwd()
        try:
            ldap_conn.extend.microsoft.modify_password(login_dn, new_password=new_passwd, old_password=None)

        except LDAPException as e:
            raise LdapServerError('Ошибка при смене пароля на УЗ: %s' % LdapServerError.err_fmt(e))

        return new_passwd
