# coding: utf-8

"""\
Логирование в системный журнал Unix

Метки в журнале о уровне сообщения:
  "`": Debug
  ".": Info
  "*": Warning
  "!": Error
  "#": Alert
"""

import syslog
from sys import exc_info
from time import time
from traceback import extract_tb, extract_stack
from datetime import timedelta

LOG_FACILITY = {
    'auth': syslog.LOG_AUTH,
    'authpriv': syslog.LOG_AUTH,
    'cron': syslog.LOG_CRON,
    'daemon': syslog.LOG_DAEMON,
    'ftp': syslog.LOG_DAEMON,
    'kern': syslog.LOG_KERN,
    'lpr': syslog.LOG_LPR,
    'mail': syslog.LOG_MAIL,
    'news': syslog.LOG_NEWS,
    'syslog': syslog.LOG_SYSLOG,
    'user': syslog.LOG_USER,
    'uucp': syslog.LOG_UUCP,
    'local0': syslog.LOG_LOCAL0,
    'local1': syslog.LOG_LOCAL1,
    'local2': syslog.LOG_LOCAL2,
    'local3': syslog.LOG_LOCAL3,
    'local4': syslog.LOG_LOCAL4,
    'local5': syslog.LOG_LOCAL5,
    'local6': syslog.LOG_LOCAL6,
    'local7': syslog.LOG_LOCAL7
}

class LoggerError(Exception): pass


# --- INTERFACE --- #
FACILITY = LOG_FACILITY['user']


def log_prep(ident, facility='user'):
    global FACILITY
    if not facility.lower() in LOG_FACILITY:
        raise LoggerError('Unknown facility')

    syslog.openlog(ident, syslog.LOG_PID)

    FACILITY = LOG_FACILITY[facility]


class Timing(object):
    def __init__(self, name=None):
        if name is None:
            self.prefix = ''
        else:
            self.prefix = '%s :: ' % name
        self.tsAll = time()
        self.ts = self.tsAll

    def getTime(self):
        return time() - self.ts

    def reset(self):
        self.ts = time()
        self.tsAll = self.ts

    def __str__(self):
        ts = time()
        return self.prefix + '%s(%.4f)' % (timedelta(seconds=(ts - self.tsAll)), ts - self.ts)

    def __call__(self, msg):
        _buf = '%s | %s' % (self, msg)
        self.ts = time()
        return _buf


class SysLogger(object):
    init_log = log_prep

    @staticmethod
    def get_timing(name=None):
        return Timing(name)

    def __init__(self, prefix, facility=FACILITY):
        self.prefix = str(prefix)
        self.facility = facility

    def _write(self, flag, mark, msg):
        for l in str(msg).splitlines():
            syslog.syslog(self.facility | flag, '%s: %s %s' % (self.prefix, mark, l))

    def __call__(self, msg):
        self._write(syslog.LOG_INFO, '.', msg)

    def err(self, msg):
        self._write(syslog.LOG_ERR, '!', msg)

    def warn(self, msg):
        self._write(syslog.LOG_WARNING, '*', msg)

    def debug(self, msg):
        self._write(syslog.LOG_DEBUG, '`', msg)

    def alert(self, msg):
        self._write(syslog.LOG_ALERT, '#', msg)

    def sublog(self, prefix):
        return self.__class__('%s/%s' % (self.prefix, prefix), self.facility)

    def excpt(self, msg, eClass=None, eObj=None, eTb=None, stack_skip=0):
        if eClass is None:
            eClass, eObj, eTb = exc_info()

        if eClass is None:
            # Если вдруг вызываем без произошедшего исключения
            self.err(msg)
        else:
            tbData_tb = list(extract_tb(eTb))[::-1]
            tbData_stack = list(extract_stack())[::-1][(2 + stack_skip):]

            self.err(msg)

            self.err('--- EXCEPTION ---')
            self.err(' %s (%s)' % (eClass.__name__, eObj))

            self.err('--- TRACEBACK ---')
            for _tbFile, _tbLine, _tbFunc, _tbText in tbData_tb:
                self.err('File: %s, line %s in %s' % (_tbFile, _tbLine, _tbFunc))
                self.err('   %s' % _tbText)

            self.err('>>> Exception Handler <<<')
            for _tbFile, _tbLine, _tbFunc, _tbText in tbData_stack:
                self.err('File: %s, line %s in %s' % (_tbFile, _tbLine, _tbFunc))
                self.err('   %s' % _tbText)

            self.err('--- END EXCEPTION ---')
