# coding: utf-8
"""\
Наполнение заданного класса данных из предоставленного словаря или объекта.

НЕ РАБОТАЕТ БЕЗ ОСТАЛЬНОГО МОДУЛЯ type_utils
"""

from dataclasses import fields, is_dataclass
from typing import Union, Dict, Any
from .type_descriptor import get_type_describer


def _dict_has(obj: Dict[str, Any], key: str) -> bool:
    return key in obj


def _dict_get(obj: Dict[str, Any], key: str) -> Any:
    return obj[key]


def _obj_has(obj: object, key: str) -> bool:
    return hasattr(obj, key)


def _obj_get(obj: object, key: str) -> Any:
    return getattr(obj, key)


def dataobj_extract(obj: Union[object, Dict[str, Any]], dataclass_type: type, type_convert: bool = False) -> object:
    """\
    Извлекает объект данных из предоставленного объекта, путём получения из него
    указанных в классе данных аттрибутов и поиска их в данном объекте.
    :param obj: Объект, из которого берутся данные для класса данных
    :param dataclass_type: Класс данных, наполняемый из ``obj``
    :param type_convert: Признак конвертирования данных. Если задан, выполняет попытку сконвертировать имеющееся
                         в параметре значение в тип, указанных в классе данных. Чудес не бывает, и процесс может
                         ошибиться особенно с Union
    """

    params = {}

    if isinstance(obj, dict):
        _has = _dict_has
        _get = _dict_get

    else:
        _has = _obj_has
        _get = _obj_get

    if not is_dataclass(dataclass_type):
        raise ValueError(f'Не относится к классам данных: {dataclass_type.__name__}')

    for fld in fields(dataclass_type):
        if _has(obj, fld.name):
            val = _get(obj, fld.name)
            type_desc = get_type_describer(fld.type)
            if val is not None:
                if not type_desc.check(val):
                    if not type_convert:
                        raise ValueError(f'Аттрибут {fld.name} не может быть получен из значения "{val}"'
                                         f' с типом {type(val).__name__} поскольку не может быть преобразован в'
                                         f' тип {type_desc}, заданный в классе данных: {type_desc.event_description}')

                    try:
                        val = type_desc(val)

                    except (ValueError, TypeError) as e:
                        raise ValueError(f'Аттрибут {fld.name} не может быть получен из значения "{val}"'
                                         f' с типом {type(val).__name__} поскольку не может быть преобразован в'
                                         f' тип {type_desc}, заданный в классе данных: {e}')

            elif not (type_desc.is_nullable or type_convert):
                raise ValueError(f'Передан "None" в поле "{fld.name}", хотя он не ожидался')

            params[fld.name] = val

    try:
        res = dataclass_type(**params)

    except (ValueError, TypeError) as e:
        _params = ', '.join(map(lambda x: f'{x[0]}="{x[1]}"', params.items()))
        raise ValueError(f'Не удалось получить объект'
                         f' класс {dataclass_type.__name__}'
                         f' из параметров: {_params}'
                         f' ошибка: {e}')

    return res


