# coding: utf-8
from configparser import ConfigParser
from dataclasses import is_dataclass, fields, Field, MISSING
from typing import Iterable, Optional, Dict, Any
from threading import RLock
from os import getenv


class ConfigParseHelperError(Exception):
    """\
    Ошибки в модуле помощника разборщика конфигураций
    """


class NoSectionNotification(ConfigParseHelperError):
    """\
    Оповещение об отсутствующей секции конфигурации
    """


class CPHSectionBase:
    """\
    Базовый класс обработки секции конфигурационного файла
    """

    def get(self, config_prop_name: str, dc_prop_name: str):
        """\
        Получить свойство из конфигурационного файла
        """
        raise NotImplemented()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        raise NotImplemented()


class CPHAbsentSection(CPHSectionBase):
    """\
    Класс создаваемый на отсутствующую секцию конфигурационного файла
    """
    def get(self, config_prop_name: str, dc_prop_name: str):
        raise NoSectionNotification()

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type == NoSectionNotification:
            return True


class CPHParamGetter(CPHSectionBase):
    def __init__(self, parser_helper_object):
        self.ph = parser_helper_object
        self.params = {}

    def _add_param(self, param_name: str, param_val: Any):
        """\
        Непосредственное добавление полученного параметра со всеми проверками.
        """

        fld = self.ph.fields.get(param_name)
        if not isinstance(fld, Field):
            raise ConfigParseHelperError(f'В классе данных отсутствует свойство "{param_name}", '
                                         f'которое мы должны заполнить из параметра конфигурации: {fld}')

        if param_val is not None:
            try:
                res = fld.type(param_val)

            except (ValueError, TypeError) as e:
                raise ConfigParseHelperError(f'При приведении параметра к '
                                             f'заданному типу произошла ошибка: '
                                             f'значение="{param_val}" ошибка="{e}"')

        else:
            if fld.default is not MISSING:
                res = fld.default

            elif fld.default_factory is not MISSING:
                res = fld.default_factory()

            else:
                raise ConfigParseHelperError('В конфигурации не заданна обязательная опция')

        self.params[param_name] = res

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None:
            self.ph.add_params(self.params)

    def get(self, config_prop_name: str, dc_prop_name: str):
        raise NoSectionNotification()


class CPHSection(CPHParamGetter):
    """\
    Класс производящий разбор конкретной секции конфигурации
    """

    def __init__(self, parser_helper_object, section: str):
        super().__init__(parser_helper_object)
        self.section_name = section
        self.section = parser_helper_object.conf_parser[section]

    def get(self, config_prop_name: str, dc_prop_name: str):
        """\
        :param config_prop_name: Имя опции в файле конфигурации
        :param dc_prop_name: Имя свойства в классе данных, хранящем конфигурацию
        """
        try:
            self._add_param(dc_prop_name, self.section.get(config_prop_name))

        except ConfigParseHelperError as e:
            raise ConfigParseHelperError(f'Ошибка при разборе параметра "{config_prop_name}" '
                                         f'в секции "{self.section_name}": {e}')


class CPHEnvParser(CPHParamGetter):
    """\
    Класс для разбора переменных окружения в том же ключе, что и файла конфигурации
    """

    def get(self, env_name: str, dc_prop_name: str):
        """\
        :param env_name: Имя переменной окружения, хранящей опцию
        :param dc_prop_name: Имя свойства в классе данных, хранящем конфигурацию
        """

        try:
            self._add_param(dc_prop_name, getenv(env_name))

        except ConfigParseHelperError as e:
            raise ConfigParseHelperError(f'Ошибка при получении значения из переменной окружения "{env_name}": {e}')


class ConfigParseHelper:
    def __init__(self, config_object_class: type, required_sections: Optional[Iterable[str]] = None):
        """\
        :param config_object_class: Dataclass, который мы подготовили как хранилище параметров конфигурации
        :param required_sections: Перечисление секций конфигурации, которые мы требуем, чтобы были в файле
        """

        if required_sections is not None:
            self.req_sections = set(required_sections)

        else:
            self.req_sections = set()

        if not is_dataclass(config_object_class):
            raise ConfigParseHelperError(f'Представленный в качестве объекта конфигурации класс не является '
                                         f'классом данных: {config_object_class.__name__}')

        self.res_obj = config_object_class
        self.fields = dict((fld.name, fld) for fld in fields(config_object_class))
        self.conf_parser: Optional[ConfigParser] = None
        self.config_params = {}
        self.config_params_lock = RLock()

    def add_params(self, params: Dict[str, Any]):
        self.config_params_lock.acquire()
        try:
            self.config_params.update(params)

        finally:
            self.config_params_lock.release()

    def section(self, section_name: str) -> CPHSectionBase:
        if self.conf_parser is None:
            raise ConfigParseHelperError(f'Прежде чем приступать к разбору файла конфигурации стоит его загрузить')

        if self.conf_parser.has_section(section_name):
            return CPHSection(self, section_name)

        else:
            return CPHAbsentSection()

    def load(self, filename: str):
        res = ConfigParser()
        try:
            res.read(filename)

        except (TypeError, IOError, OSError, ValueError) as e:
            raise ConfigParseHelperError(f'Не удалось загрузить файл конфигурации: файл="{filename}" '
                                         f'ошибка="{e}"')

        missing_sections = self.req_sections - set(res.sections())

        if missing_sections:
            missing_sections = ', '.join(missing_sections)
            raise ConfigParseHelperError(f'В конфигурационном файле отсутствуют секции: {missing_sections}')

        self.conf_parser = res

    def get_config(self):
        try:
            return self.res_obj(**self.config_params)

        except (ValueError, TypeError) as e:
            raise ConfigParseHelperError(f'Не удалось инициализировать объект конфигурации: {e}')
