# coding: utf-8
import ldap

# ===========================================================
# Статус завершения операций
STATUS_OK = 0           # Всё ОК.
STATUS_BADAUTH = 1      # Не верные пользователь или пароль.
STATUS_SERVERDOWN = 2   # Сервер не доступен.
STATUS_SERVERERROR = 3  # Ошибка при взаимодействии с сервером.


class LdapError(Exception):
    pass


class LdapAuth(object):
    def __init__(self, server, user_prefix='CORP\\', base_dn='DC=example,DC=net', attr=None):
        self.server = server
        self.prefix = user_prefix
        self.status = ''
        self.baseDN = base_dn
        self.status_code = 0
        self.groups = None
        self.attr = None
        self._needAttr = [i for i in map(str, attr)] if attr is not None else []

    def __call__(self, user, passwd):
        conn = None
        try:
            conn = ldap.initialize(self.server)
            conn.set_option(ldap.OPT_REFERRALS, 0)
            conn.simple_bind_s(self.prefix + user, passwd)

        except ldap.INVALID_CREDENTIALS:
            if conn is not None:
                conn.unbind()

            self.status = 'Invalid credentials'
            self.status_code = STATUS_BADAUTH
            return False

        except ldap.SERVER_DOWN:
            self.status = 'Server is down'
            self.status_code = STATUS_SERVERDOWN
            return False

        self.groups = []
        try:
            ldap_data = conn.search_s(self.baseDN, ldap.SCOPE_SUBTREE, f'(cn={user})',
                                      ['memberOf'] + self._needAttr)[0][1]

            for i in ldap_data['memberOf']:
                self.groups.append(i.split(',')[0].split('=')[1].decode('utf-8'))

            del ldap_data['memberOf']
            self.attr = ldap_data

        except KeyError:
            self.status = 'User object from LDAP is wrong, it can be anonymous logon'
            self.status_code = STATUS_SERVERERROR
            return False

        finally:
            conn.unbind()

        return True

    def __getitem__(self, key):
        return self.attr[key]

    def member_of(self, group):
        """Проверка на присутствие у пользователя некоторой группы
        """
        if self.groups is None:
            raise LdapError('Request membership before auth call')

        if not isinstance(group, unicode):
            if isinstance(group, str):
                group = group.decode('utf-8')
            else:
                group = str(group).decode('utf-8')

        if group in self.groups:
            return True
        else:
            return False

    def __contains__(self, group):
        return self.member_of(group)

    def member_of_groups(self, groups):
        if not len(groups) > 0:
            return False

        for group in groups:
            if group not in self:
                return False

        return True
