# coding: utf-8
"""\
Набор инструментов, для более удобного взаимодействия с фреймворком ``bottle``
"""

import bottle
import json
from typing import Union, List, Optional, Tuple, Any
from datetime import datetime
from hashlib import sha512
from dataclasses import is_dataclass, asdict


VARIABLE_PREFIX = 'ru.a0fs.app'     # Префикс переменных и иных структур, которые сохраняются во внутренних
                                    # структурах ``bottle``
IP_HEADER = 'X-Real-IP'             # Заголовок, в котором реверс-прокси хранит реальный IP клиента.
ID_HEADER = 'X-Req-ID'              # Заголовок запроса, в котором может храниться уникальный ID запроса,
                                    # выставленного реверс-прокси.
CONN_ID_HEADER = 'X-Conn-ID'        # Идентификатор текущего соединения.


class Cookie(object):
    """\
    Класс хранящий ``cookie`` и способный их устанавливать в объекты http-ответов ``bottle``
    """

    def __init__(self, name: str, value: str,
                 max_age: int = None, expires: Union[int, datetime] = None, path: str = None,
                 secure: bool = True, httponly: bool = True,
                 samesite: bool = False, domain: str = None):
        """\
        :param name: Имя ``cookie``
        :param value: Значение ``cookie``
        :param max_age: Время жизни ``cookie`` в секундах.
        :param expires: Значение времени, когда cookie будет удалена, задаётся в виде ``unix timestamp (int)`` или
                        ``datetime``
        :param path: Префикс пути поиска ресурса на данном сайте, для которого следует отправлять данное ``cookie``
        :param secure: Отправлять ``cookie`` только по шифрованным каналам связи
        :param httponly: Сделать ``cookie`` не доступной для ``JavaScript``
        :param samesite: Не отправлять данную cookie, если запрос пришёл не с того же сайта (анализируется заголовок
                         referer)
        :param domain: Имя домена в рамках которого выставляется cookie. В современных браузерах может и глючить
        """

        self.name = name
        self.value = value
        self.max_age = max_age
        self.expires = expires
        self.path = path
        self.secure = secure
        self.httponly = httponly
        self.samesite = samesite
        self.domain = domain

    def response_add(self, resp: bottle.BaseResponse):
        res = {
            'name': self.name,
            'value': self.value,
            'secure': self.secure,
            'httponly': self.httponly,
        }

        if self.samesite:
            res['samesite'] = 'strict'

        for k in ('max_age', 'expires', 'path', 'domain'):
            if getattr(self, k) is not None:
                res[k] = getattr(self, k)

        resp.set_cookie(**res)


def get_client_ip() -> str:
    """\
    Получение реального адреса клиента
    """

    ip = bottle.request.get_header(IP_HEADER)
    if ip:
        return ip

    else:
        return bottle.request.remote_addr


def get_session_fingerprint(*add_params) -> str:
    """\
    Конструируем некий отпечаток пользовательской сессии.

    В качестве параметра принимается всё, что должно участвовать в создании отпечатка.
    Это всё превращается в строки и подмешивается в хэш
    """

    ua = bottle.request.get_header('user-agent')
    add_params_mixin = ':'.join(map(str, add_params))

    return sha512(f'{ua}:{get_client_ip()}:{add_params_mixin}'.encode('utf-8')).hexdigest().lower()


def variable_prep(value, new_value_type: type = str, postprocess: bool = True) -> Any:
    """\
    Более интеллектуальная процедура преобразования базовых типов

    :param value: Значение, которое необходимо преобразовать
    :param new_value_type: Тип в который необходимо преобразовать значение
    :param postprocess: Применять ли последующую обработку полученного результата
    :returns: Значение ``value`` преобразованное к типу ``value_type``
    """

    if value is None:
        return value

    if issubclass(new_value_type, bool):
        if isinstance(value, str):
            if value.lower() in ('on', 'yes', '1', 'true',):
                return True
            elif value.lower() in ('off', 'no', '0', 'false',):
                return False
            else:
                raise ValueError(f'Unknown method translate "{value}" to "{new_value_type.__name__}"')

    res = new_value_type(value)

    if isinstance(res, str) and postprocess:
        res = res.strip()

    return res


def get_variable(name: str, default=None):
    """\
    Возвращает значения из хранилища контекста запроса bottle
    """

    _name = f'{VARIABLE_PREFIX}.{name}'
    if _name in bottle.request.environ:
        return bottle.request.environ[_name]

    else:
        return default


def set_variable(name: str, value):
    """\
    Устанавливает значения в хранилище контекста запроса bottle
    """

    bottle.request.environ[f'{VARIABLE_PREFIX}.{name}'] = value


def make_response(
        data=None, status=200, cookies: Optional[List[Cookie]] = None
) -> bottle.HTTPResponse:
    """\
    Формирует ``API`` ответ.
    """

    if data is None:
        data = {}

    res = bottle.HTTPResponse(body=json.dumps(json_type_sanitizer(data)), status=status)
    res.content_type = 'application/json'

    if cookies is not None:
        for cookie in cookies:
            cookie.response_add(res)

    return res


def get_request_json() -> Optional[dict]:
    """\
    Если в запросе, который мы обрабатываем в текущий момент, использовалась посылка данных JSON, получить их.
    """

    res = bottle.request.json
    if res is None:
        raise ValueError('Не обнаружено JSON в запросе')

    return res


def get_cookie(name: str) -> Optional[str]:
    """\
    Получить значение ``cookie`` с заданным именем из текущего запроса.
    """

    return bottle.request.get_cookie(name)


def get_param(name: str, param_type: Union[type, str] = str,
              default=None, postprocess: bool = True, param_source: str = None):
    """\
    Получить из обрабатываемого на данный момент запроса значения установленного параметра. Обрабатываются как
    ``GET``, так и ``POST`` параметры, если иное не указано конкретно.

    :param name: Имя параметра
    :param param_type: Тип, в который нужно преобразовать полученный параметр
    :param default: Значение, отдаваемое, если параметр не установлен в запросе
    :param postprocess: Производить ли постобработку параметра
    :param param_source: Источник параметра, [``get``, ``post``]
    :returns Полученное из запроса значение
    """

    if param_source is None:
        res = bottle.request.params.getunicode(name)
    elif param_source.lower() == 'get':
        res = bottle.request.GET.getunicode(name)
    elif param_source.lower() == 'post':
        res = bottle.request.POST.getunicode(name)
    else:
        raise ValueError(f'Unknown parameter source "{param_source}" for "{name}"')

    if res is None:
        res = default

    if isinstance(param_type, str):
        if param_type.lower() == 'json':
            if not res:
                return None

            else:
                try:
                    return json.loads(res)

                except json.JSONDecodeError as e:
                    raise ValueError(f'Unable to get JSON from from parameter "{name}": param="{res}" error="{e}"')
    else:
        try:
            return variable_prep(res, new_value_type=param_type, postprocess=postprocess)

        except (TypeError, ValueError) as e:
            raise ValueError(f'Error parsing parameter "{name}": {e}')


def dict_filter(data: dict, need_keys: Union[List[str], Tuple[str]]) -> dict:
    """\
    Создать новый словарь на основе существующего, добавив в него только нужные ключи
    """

    return dict((key, val) for key, val in data.items() if key in need_keys)


def json_type_sanitizer(val):
    """\
    Преобразует значение ``val`` в пригодное для преобразования в json значение.
    """

    val_t = type(val)

    if is_dataclass(val):
        return json_type_sanitizer(asdict(val))

    elif val_t in (int, float, str, bool) or val is None:
        return val

    elif val_t in (list, tuple):
        return list(map(json_type_sanitizer, val))

    elif val_t == dict:
        return dict((key, json_type_sanitizer(d_val)) for key, d_val in val.items())

    else:
        return str(val)


def make_log_ident(user: Optional[str] = None) -> str:
    """\
    Создаём строку, идентифицирующую данный запрос для журналирования операций.

    :param user: Если известен пользователь, задаём его здесь
    """

    ip = get_client_ip()
    url_path = bottle.request.fullpath
    conn_id = bottle.request.get_header(CONN_ID_HEADER)

    if conn_id is not None:
        ip = f'{ip} | {conn_id}'

    if user is not None:
        ip = f'_NOID_[{ip}]'

    else:
        ip = f'{user}[{ip}]'

    return f'{ip} - {url_path}'


def make_error_response(
        code: int, err: Union[Exception, str],
        msg: str = None,
        remove_cookies: Optional[List[str]] = None,
        cookies: Optional[List[Cookie]] = None
) -> bottle.HTTPResponse:
    """\
    Создание сообщения об ошибке для JSON REST API сервисов

    :param code: HTTP-код ответа
    :param err: Либо объект исключения, либо название ошибки
    :param msg: Если не ``None`` - сообщение об ошибке. В противном случае, если ``err`` является исключением,
                сообщение берётся из ``str(err)`` если нет, становится пустой строкой.
    :param remove_cookies: Список cookie, которые должны быть удалены с клиента.
    :param cookies: Набор cookies вставляемый в конструктор ответа без изменений. Читать в соответствующем
                    параметре ``make_response``
    """

    if isinstance(err, Exception):
        err_type = type(err).__name__
        err_msg = str(err) if msg is None else msg

    else:
        err_type = err
        err_msg = msg if msg is not None else ''

    res = make_response({
        'error': err_type,
        'msg': err_msg,
    }, code, cookies)

    if remove_cookies is not None:
        for cookie_name in remove_cookies:
            res.delete_cookie(cookie_name)

    return res
