# coding: utf-8
""" Логирование на консоль

Метки в журнале о уровне сообщения:
  "`": Debug
  ".": Info
  "*": Warning
  "!": Error
  "#": Alert

"""

from time import monotonic, ctime
from datetime import timedelta, date
from traceback import extract_tb, extract_stack
from sys import exc_info, stderr, stdout
from typing import Optional, TextIO, Any
from os.path import join as p_join, abspath
from threading import RLock


TIME_TO_FLUSH: int = 120    # Время по умолчанию с момента прошлого сброса лога,
                            # после которого, выполняется принудительный сброс буферов


class Timing(object):
    def __init__(self, name: Optional[str] = None):
        if name is None:
            self.prefix = ''

        else:
            self.prefix = f'{name} :: '

        self.tsAll = monotonic()
        self.ts = self.tsAll

    def get_time(self):
        return monotonic() - self.ts

    def reset(self):
        self.ts = monotonic()
        self.tsAll = self.ts

    def __str__(self):
        ts = monotonic()
        return self.prefix + '%s(%.4f)' % (timedelta(seconds=(ts - self.tsAll)), ts - self.ts)

    def __call__(self, msg):
        _buf = f'{self} | {msg}'
        self.ts = monotonic()
        return _buf


class NullLog(object):
    def __init__(self, prefix: str = 'main'):
        self.prefix = prefix

    @staticmethod
    def _write(mark: str, msg: Any):
        pass     # cat > /dev/null

    def __call__(self, msg):
        self._write('.', msg)

    def err(self, msg):
        self._write('!', msg)

    def warn(self, msg):
        self._write('*', msg)

    def alert(self, msg):
        self._write('#', msg)

    def debug(self, msg):
        self._write('`', msg)

    @staticmethod
    def get_timing(name: Optional[str] = None):
        return Timing(name)

    def sub_log(self, name: str):
        return self.__class__(f'{self.prefix}/{name}')

    def excpt(self, msg, e_class=None, e_obj=None, e_tb=None, stack_skip=0):
        if e_class is None:
            e_class, e_obj, e_tb = exc_info()

        tb_data_tb = list(extract_tb(e_tb))[::-1]
        tb_data_stack = list(extract_stack())[::-1][(2 + stack_skip):]
        self.err(msg)
        self.err('--- EXCEPTION ---')
        self.err(f' {e_class.__name__} ({e_obj})')
        self.err('--- TRACEBACK ---')
        for _tb_file, _tb_line, _tb_func, _tb_text in tb_data_tb:
            self.err(f'File: {_tb_file}, line {_tb_line} in {_tb_func}')
            self.err(f'   {_tb_text}')

        self.err('>>> Exception Handler <<<')
        for _tb_file, _tb_line, _tb_func, _tb_text in tb_data_stack:
            self.err(f'File: {_tb_file}, line {_tb_line} in {_tb_func}')
            self.err(f'   {_tb_text}')

        self.err('--- END EXCEPTION ---')


class FileLog(NullLog):
    @staticmethod
    def _open_file(file_name: str):
        return open(file_name, 'a', encoding='utf-8')

    def __init__(self, prefix: str = 'main',
                 file_name: Optional[str] = None,
                 file_obj: Optional[TextIO] = None,
                 time_to_flush: int = TIME_TO_FLUSH
                 ):

        super().__init__(prefix=prefix)
        self.fd: Optional[TextIO] = None

        self.flush_time = monotonic()
        self.time_to_flush = time_to_flush      # Время с момента прошлого сброса лога,
                                                # после которого, выполняется принудительный сброс буферов

        if file_name is not None:
            self.fd = self._open_file(file_name)

        else:
            self.fd = file_obj

        if self.fd is None:
            raise ValueError(f'Не задан файл для записи журналов')

    def flush(self, time_mark: float = None):
        if time_mark is None:
            time_mark = monotonic()

        self.flush_time = time_mark
        self.fd.flush()

    def close(self):
        self.fd.flush()
        self.fd.close()
        self.fd = None

    def __del__(self):
        if self.fd is not None:
            self.fd.flush()
            self.fd.close()
            self.fd = None

    def flush_event(self):
        t = monotonic()
        if t - self.flush_time >= self.time_to_flush:
            self.flush(t)

    def _write(self, mark: str, msg: Any):
        if self.fd is None:
            raise ValueError('Попытка использовать закрытый файл журнала')

        t = ctime()
        for l in str(msg).splitlines():
            self.fd.write(f'{t} | {mark} {self.prefix} | {l}')

        self.flush_event()

    def sub_log(self, name: str):
        if self.fd is None:
            raise ValueError('Попытка использовать закрытый файл журнала')

        return self.__class__(f'{self.prefix}/{name}', file_obj=self.fd, time_to_flush=self.time_to_flush)


class StderrLog(FileLog):
    def __init__(self, prefix: str = 'main'):
        super().__init__(prefix, file_obj=stderr)

    def flush_event(self):
        pass     # нет необходимости сбрасывать буферы в консоли


class StdoutLog(FileLog):
    def __init__(self, prefix: str = 'main'):
        super().__init__(prefix, file_obj=stdout)

    def flush_event(self):
        pass     # нет необходимости сбрасывать буферы в консоли


class LogrotateFile(FileLog):
    def __init__(self, directory: str = '.', prefix: str = 'main', time_to_flush: int = TIME_TO_FLUSH):
        d = date.today()
        directory = abspath(directory)
        file_name = p_join(directory, f'{prefix}-{d}.log')

        super().__init__(prefix, file_name=file_name, time_to_flush=time_to_flush)
        self.logrotate_base = p_join(directory, f'{prefix}')
        self.logrotate_date = d

    def flush(self, time_mark: float = None, no_rotate: bool = False):
        if not no_rotate:
            d = date.today()
            if self.logrotate_date != d:
                self.logrotate_rotate()
                return

        super().flush(time_mark=time_mark)

    def logrotate_rotate(self):
        d = date.today()
        file_name = f'{self.logrotate_base}-{d}.log'
        self.fd.flush()
        self.fd = self._open_file(file_name)
        self.logrotate_date = d
        self.flush(no_rotate=True)


class ThreadSafeFileLog(FileLog):
    def __int__(self, *a, **kwa):
        super().__init__(*a, **kwa)
        self.thread_safe_lock = RLock()

    def _write(self, mark: str, msg: Any):
        with self.thread_safe_lock:
            super()._write(mark, msg)

    def flush(self, time_mark: float = None):
        with self.thread_safe_lock:
            super().flush(time_mark)


class ThreadSafeLogrotateFile(LogrotateFile):
    def __init__(self, directory: str = '.', prefix: str = 'main', time_to_flush: int = TIME_TO_FLUSH):
        super().__init__(directory=directory, prefix=prefix, time_to_flush=time_to_flush)
        self.thread_safe_lock = RLock()

    def _write(self, mark: str, msg: Any):
        with self.thread_safe_lock:
            super()._write(mark, msg)

    def flush(self, time_mark: float = None, no_rotate: bool = False):
        with self.thread_safe_lock:
            super().flush(time_mark=time_mark, no_rotate=no_rotate)

    def logrotate_rotate(self):
        with self.thread_safe_lock:
            super().logrotate_rotate()
