# coding: utf-8

from ldap3 import Server, Connection, SIMPLE, SUBTREE, IP_V4_PREFERRED
from ldap3.core.exceptions import LDAPException, LDAPInvalidCredentialsResult
from ldap3.utils.conv import escape_filter_chars
from collections import namedtuple
from typing import List, Optional, Dict, Union


class LdapError(Exception):
    pass


class LdapServerError(LdapError):
    pass


class LdapUserError(LdapError):
    pass


class LdapNotFound(LdapUserError):
    def __init__(self, filter: str, attribs: List[str]):
        _attribs = ', '.join(attribs)
        super().__init__(f'Данные согласно фильтру в LDAP не найдены: {filter}, attrib={_attribs}')


class LdapAuthDenied(LdapUserError):
    def __init__(self, user):
        super(LdapAuthDenied, self).__init__(f'Авторизация пользователя "{user}" отклонена сервером')


LdapUserInfo = namedtuple('LdapUserInfo', [
    'name', 'mail', 'groups',
])

ACC_STATUS_ACCOUNTDISABLE = 2
ACC_STATUS_LOCKOUT = 16
ACC_STATUS_PASSWORD_EXPIRED = 8388608
ACC_STATUS_NORMAL_ACCOUNT = 512
ACC_STATUS_MASK = ACC_STATUS_ACCOUNTDISABLE | ACC_STATUS_LOCKOUT \
                   | ACC_STATUS_PASSWORD_EXPIRED | ACC_STATUS_NORMAL_ACCOUNT

ACC_STATUS_FAIL_MASK = ACC_STATUS_ACCOUNTDISABLE | ACC_STATUS_LOCKOUT | ACC_STATUS_PASSWORD_EXPIRED


class LdapRes(object):
    def __init__(self, dn: str, attribs: Dict[str, Union[str, List[str]]]):
        self.dn = dn
        self.attribs = attribs

    def __str__(self):
        return self.dn


class LdapConfig(object):
    def __init__(self,
                 server_uri: str,
                 user_name: str, passwd: str,
                 base_dn: Optional[str] = None,
                 timeout: int = 150, reconnects: int = 3,
                 auth_domain: str = ''
                 ):

        self.server_uri = server_uri   # URL Доступа к северу: 'ldap://servername:port/'
        self.user_name = user_name     # Имя пользователя для подключения к серверу LDAP
        self.passwd = passwd           # Пароль для подключения к серверу LDAP
        self.base_dn = base_dn         # База поиска
        self.timeout = timeout         # Выставляемые таймауты на операцию
        self.reconnects = reconnects   # Количество попыток переподключиться
        self.auth_domain = auth_domain # Имя домена авторизации

    def __str__(self):
        return f'{self.user_name}@{self.server_uri} ({self.base_dn} timeout="{self.timeout}" auth_domain="{self.auth_domain}")'


class Ldap(object):
    escape_filter_chars = escape_filter_chars

    def __init__(self, config: LdapConfig):
        self.server = Server(config.server_uri, connect_timeout=config.timeout, mode=IP_V4_PREFERRED)
        self.config = config

    def get_connection(self) -> Connection:
        _reconnects = 1

        # Подготавливаем имя пользователя к авторизации в AD
        if self.config.auth_domain != '':
            _ldap_auth_uname = f'{self.config.auth_domain}\\{self.config.user_name}'
        else:
            _ldap_auth_uname = self.config.user_name

        while True:
            _reconnects += 1
            try:
                ldap_connection = Connection(self.server, authentication=SIMPLE,
                                             user=_ldap_auth_uname, password=self.config.passwd,
                                             check_names=True,
                                             auto_referrals=False, raise_exceptions=True, auto_range=True,
                                             )

                ldap_connection.open()
                if not ldap_connection.bind():
                    continue  # Пытаемся переподключиться при ошибках

                break

            except LDAPInvalidCredentialsResult:
                raise LdapAuthDenied(self.config.user_name)

            except LDAPException as e:
                if _reconnects > self.config.reconnects:
                    # Возбуждаем исключение после того как попробовали несколько раз
                    raise LdapServerError(f'Ошибка подключения к серверу: {e}')

        if ldap_connection is None:
            raise LdapServerError('Не выполнено подключение к серверу')

        return ldap_connection

    def search(self, ldap_filter: str,
               attribs: Optional[List[str]] = None,
               base_dn: Optional[str] = None):

        if base_dn is None:
            if not self.config.base_dn:
                raise LdapUserError('Не задан Base DN для поиска в LDAP')

            base_dn = self.config.base_dn

        if attribs is None:
            attribs = ['cn']

        ldap_connection = self.get_connection()
        ldap_connection.search(base_dn, ldap_filter, attributes=attribs)
        if ldap_connection.result['result'] != 0:
            raise LdapServerError(f'Не могу выполнить запрос к серверу LDAP: '
                                  f'{ldap_connection.result.get("description")}')

        for itm in ldap_connection.response:
            yield LdapRes(dn=itm['dn'], attribs=itm['attributes'])

    @staticmethod
    def filter_group_names(group):
        group_res = list(filter(lambda x: x.lower().startswith('cn'), group.split(',')))
        if group_res:
            try:
                return group_res[0].split('=')[1]
            except IndexError:
                raise LdapError(f'Имя группы с неожиданным форматом: {group_res[0]}')
        else:
            raise LdapError(f'Нет атрибута CN в имени: {group}')


    @staticmethod
    def decode_acc_status(acc_status):
        res = []

        if acc_status & ACC_STATUS_ACCOUNTDISABLE != 0:
            res.append('ACCOUNTDISABLE')

        if acc_status & ACC_STATUS_NORMAL_ACCOUNT != 0:
            res.append('NORMAL_ACCOUNT')

        if acc_status & ACC_STATUS_LOCKOUT != 0:
            res.append('LOCKOUT')

        if acc_status & ACC_STATUS_PASSWORD_EXPIRED != 0:
            res.append('PASSWORD_EXPIRED')

        return res
