# coding: utf-8
"""
Тривиальная реализация сислоггера без блэк-джека и поэтесс

Метки в журнале о уровне сообщения:
  "`": Debug
  ".": Info
  "*": Warning
  "!": Error
  "#": Alert

"""
from time import monotonic
from datetime import timedelta

import syslog
from traceback import extract_tb, extract_stack
from sys import exc_info


class Timing(object):
    def __init__(self, name=None):
        if name is None:
            self.prefix = ''
        else:
            self.prefix = '%s :: ' % name
        self.tsAll = monotonic()
        self.ts = self.tsAll

    def getTime(self):
        return monotonic() - self.ts

    def reset(self):
        self.ts = monotonic()
        self.tsAll = self.ts

    def __str__(self):
        ts = monotonic()
        return self.prefix + '%s(%.4f)' % (timedelta(seconds=(ts - self.tsAll)), ts - self.ts)

    def __call__(self, msg):
        _buf = '%s | %s' % (self, msg)
        self.ts = monotonic()
        return _buf


class SimpleSysLogger(object):
    @staticmethod
    def init_syslog(ident):
        syslog.openlog(ident, syslog.LOG_PID)

    @staticmethod
    def get_timing(name=None):
        return Timing(name)

    def __init__(self, prefix='main', facility=syslog.LOG_USER):
        self.prefix = str(prefix)
        self.facility = facility

    def _write(self, flag, mark, msg):
        for l in str(msg).splitlines():
            syslog.syslog(self.facility | flag, f'{self.prefix}: {mark} {l}')

    def __call__(self, msg):
        self._write(syslog.LOG_INFO, '.', msg)
    
    def err(self, msg):
        self._write(syslog.LOG_ERR, '!', msg)
    
    def warn(self, msg):
        self._write(syslog.LOG_WARNING, '*', msg)
    
    def debug(self, msg):
        self._write(syslog.LOG_DEBUG, '`', msg)

    def alert(self, msg):
        self._write(syslog.LOG_ALERT, '#', msg)

    def sub_log(self, prefix):
        return self.__class__('%s/%s' % (self.prefix, prefix), self.facility)

    def excpt(self, msg, e_class=None, e_obj=None, e_tback=None, stack_skip=0):
        if e_class is None:
            e_class, e_obj, e_tback = exc_info()
        
        tb_data_tb = list(extract_tb(e_tback))[::-1]
        tb_data_stack = list(extract_stack())[::-1][(2 + stack_skip):]

        self.err(msg)
        self.err('--- EXCEPTION ---')
        self.err(' %s (%s)' % (e_class.__name__, e_obj))

        self.err('--- TRACEBACK ---')
        for _tb_file, _tb_line, _tb_func, _tb_text in tb_data_tb:
            self.err('File: %s, line %s in %s' % (_tb_file, _tb_line, _tb_func))
            self.err('   %s' % _tb_text)

        self.err('>>> Exception Handler <<<')
        for _tb_file, _tb_line, _tb_func, _tb_text in tb_data_stack:
            self.err('File: %s, line %s in %s' % (_tb_file, _tb_line, _tb_func))
            self.err('   %s' % _tb_text)

        self.err('--- END EXCEPTION ---')
