# coding: utf-8

"""\
Логирование в системный журнал Unix

Метки в журнале о уровне сообщения:
  "`": Debug
  ".": Info
  "*": Warning
  "!": Error
  "#": Alert
"""

import syslog
from sys import exc_info
from time import monotonic
from traceback import extract_tb, extract_stack
from datetime import timedelta
from typing import Any

LOG_FACILITY = {
    'auth': syslog.LOG_AUTH,
    'authpriv': syslog.LOG_AUTH,
    'cron': syslog.LOG_CRON,
    'daemon': syslog.LOG_DAEMON,
    'ftp': syslog.LOG_DAEMON,
    'kern': syslog.LOG_KERN,
    'lpr': syslog.LOG_LPR,
    'mail': syslog.LOG_MAIL,
    'news': syslog.LOG_NEWS,
    'syslog': syslog.LOG_SYSLOG,
    'user': syslog.LOG_USER,
    'uucp': syslog.LOG_UUCP,
    'local0': syslog.LOG_LOCAL0,
    'local1': syslog.LOG_LOCAL1,
    'local2': syslog.LOG_LOCAL2,
    'local3': syslog.LOG_LOCAL3,
    'local4': syslog.LOG_LOCAL4,
    'local5': syslog.LOG_LOCAL5,
    'local6': syslog.LOG_LOCAL6,
    'local7': syslog.LOG_LOCAL7
}


class LoggerError(Exception): pass


# --- INTERFACE --- #
FACILITY = LOG_FACILITY['user']


def check_facility(facility: str = 'user'):
    if not facility.lower() in LOG_FACILITY:
        raise LoggerError(f'Unknown facility: {facility}')


def log_prep(ident: str, facility: str = 'user'):
    global FACILITY
    check_facility(facility)

    syslog.openlog(ident, syslog.LOG_PID)

    FACILITY = LOG_FACILITY[facility]


class Timing(object):
    def __init__(self, name: str = None):
        if name is None:
            self.prefix = ''
        else:
            self.prefix = f'{name} :: '
        self.tsAll = monotonic()
        self.ts = self.tsAll

    def get_time(self):
        return monotonic() - self.ts

    def reset(self):
        self.ts = monotonic()
        self.tsAll = self.ts

    def __str__(self):
        ts = monotonic()
        return self.prefix + '%s(%.4f)' % (timedelta(seconds=(ts - self.tsAll)), ts - self.ts)

    def __call__(self, msg: Any) -> str:
        _buf = f'{self} | {msg}'
        self.ts = monotonic()
        return _buf


class SysLogger(object):
    init_log = log_prep

    @staticmethod
    def get_timing(name: str = None):
        return Timing(name)

    def __init__(self, prefix: str = 'main', facility: int = FACILITY):
        self.prefix = str(prefix)
        self.facility = facility

    def _write(self, flag: int, mark: str, msg: Any):
        for l in str(msg).splitlines():
            syslog.syslog(self.facility | flag, f'{mark} {self.prefix}: {l}')

    def __call__(self, msg: Any):
        self._write(syslog.LOG_INFO, '.', msg)

    def err(self, msg: Any):
        self._write(syslog.LOG_ERR, '!', msg)

    def warn(self, msg: Any):
        self._write(syslog.LOG_WARNING, '*', msg)

    def debug(self, msg: Any):
        self._write(syslog.LOG_DEBUG, '`', msg)

    def alert(self, msg: Any):
        self._write(syslog.LOG_ALERT, '#', msg)

    def sublog(self, prefix: str):
        return self.__class__(f'{self.prefix}/{prefix}', self.facility)

    def excpt(self, msg: Any, e_class: type = None, e_obj: Exception = None, e_tb=None, stack_skip=0):
        if e_class is None:
            e_class, e_obj, e_tb = exc_info()

        if e_class is None:
            # Если вдруг вызываем без произошедшего исключения
            self.err(msg)
        else:
            tb_data_tb = list(extract_tb(e_tb))[::-1]
            tb_data_stack = list(extract_stack())[::-1][(2 + stack_skip):]

            self.err(msg)

            self.err('--- EXCEPTION ---')
            self.err(' %s (%s)' % (e_class.__name__, e_obj))

            self.err('--- TRACEBACK ---')
            for _tb_file, _tb_line, _tb_func, _tb_text in tb_data_tb:
                self.err('File: %s, line %s in %s' % (_tb_file, _tb_line, _tb_func))
                self.err('   %s' % _tb_text)

            self.err('>>> Exception Handler <<<')
            for _tb_file, _tb_line, _tb_func, _tb_text in tb_data_stack:
                self.err('File: %s, line %s in %s' % (_tb_file, _tb_line, _tb_func))
                self.err('   %s' % _tb_text)

            self.err('--- END EXCEPTION ---')
