# coding: utf-8
"""
Смена пароля на пользователя в AD посредством Python

"""

from ldap3 import Server, Connection, SIMPLE, SUBTREE
from ldap3.core.exceptions import LDAPException
from secrets import token_urlsafe
import string

class LdapError(Exception):
    @staticmethod
    def errFmt(e):
        _buf = ' '.join(str(e).split())            # всё в одну строку с одним пробелом
                                                   # между словами
        return "%s(%s)" % (type(e).__name__, _buf)
        
class LdapServerError(LdapError): pass
class LdapPassGenError(LdapError): pass
class LdapNotFound(LdapError): pass


class LdapPasswdChanger(object):
    def __init__(self, ldap_server_uri, ldap_user, ldap_passwd, ldap_baseDN, ldap_timeout=120, ldap_reconnect=5, passwd_len=25):
        self.ldap_server_uri = ldap_server_uri  # URI к серверу: ldaps://server_name:port_number
        self.ldap_user = ldap_user              # Пользователь, имеющий право сброса паролей
        self.ldap_passwd = ldap_passwd          # Пароль к тому пользователю
        self.ldap_baseDN = ldap_baseDN
        self.ldap_timeout = ldap_timeout        # Таймаут операций с сервером
        self.passwd_len = passwd_len            # Желаемая длинна пароля
        self.ldap_reconnect = ldap_reconnect    # Количество попыток переподключения к серверу

        # Заготовка под пароли
        self._set_punct = set(string.punctuation)
        self._set_uper = set(string.ascii_uppercase)
        self._set_lower = set(string.ascii_lowercase)

    def getPasswd(self):
        _cnt = 0
        while True:
            res = token_urlsafe(self.passwd_len)
            _res_set = set(res)
            if ((_res_set & self._set_punct)
                and (_res_set & self._set_uper)
                and (_res_set & self._set_lower)
            ):
                return res

            _cnt += 1
            if _cnt > 60:
                raise LdapPassGenError('Не удалось сгенерировать пароль')


    def changePass(self, login):
        _reconnects = self.ldap_reconnect
        if _reconnects is None or _reconnects < 1:
            _reconnects = 1

        ldapConn = None
        while True:
            _reconnects -= 1
            try:
                ldapSrv = Server(self.ldap_server_uri, connect_timeout=self.ldap_timeout)
                ldapConn = Connection(ldapSrv, authentication=SIMPLE,
                                      user=self.ldap_user, password=self.ldap_passwd,
                                      check_names=True,
                                      auto_referrals=False, raise_exceptions=True, auto_range=True,
                                      )
                ldapConn.open()
                if not ldapConn.bind():
                    continue

                break
            except LDAPException as e:
                if not _reconnects > 0:
                    raise LdapServerError('Ошибка подключения к серверу: %s' % LdapServerError.errFmt(e))

        filter = '(sAMAccountName=%s)' % login
        try:
            if not ldapConn.search(self.ldap_baseDN, filter):
                raise LdapServerError('Сервер вернул статус ощибки: %(desc)s(%(code)s) - %(msg)s' % {
                    'desc': ldapConn.result['description'],
                    'code': ldapConn.result['result'],
                    'msg': ldapConn.result['message'],
                    })
        except LDAPException as e:
            raise LdapServerError('Ошибка при поиска УЗ в ldap: %s' % LdapServerError.errFmt(e))

        if not len(ldapConn.entries) > 0:
            raise LdapNotFound('Не найдена учётная запись: %s' % login)
        elif len(ldapConn.entries) != 1:
            raise LdapError('Наёдено более одной УЗ по заданному имени %s: %s - %s' %(
                login,
                len(ldapConn.entries),
                '; '.join([i.entry_dn for i in ldapConn.entries[:10]])
            ))

        login_dn = ldapConn.entries[0].entry_dn
        new_passwd = self.getPasswd()
        try:
            ldapConn.extend.microsoft.modify_password(login_dn, new_password=new_passwd, old_password=None)
        except LDAPException as e:
            raise LdapServerError('Ошибка при смене пароля на УЗ: %s' % LdapServerError.errFmt(e))

        return new_passwd

    
    
    

