# -*- coding: utf-8 -*-
import paramiko
from datetime import datetime

# --- OPTS --- #
TIMEOUT = 120

# --- CONST --- #
# Политика ключей узла
HOSTKEY_autoAdd = 0
HOSTKEY_warn = 1
HOSTKEY_reject = 2

def promptChecker(user, host, buf):
	Signs = ['[', ']', '#', '<', '>', '$', '!', '%', '(', ')', '@', '+', ':', ';', ' ', '\t']
	isPrompt = False
	isSep = True
	newLites = 0
	endBuf = ''
	for i in range(1, len(buf) + 1):
		if not isPrompt:
			if buf[-i] == '\n':
				endBuf = buf[-i] + endBuf
				continue
			isPrompt = True
		if isSep:
			if buf[-i] in Signs:
				endBuf = buf[-i] + endBuf
				continue
			isSep = False
		if buf[-i] == '\n':
			break
	_startProc = len(buf) - i
	_endProc = len(buf) - len(endBuf)
	buf = buf[_startProc:_endProc]
	searchCond = []
	for i in [user, host]:
		bufL = ''
		bufR = ''
		index = buf.find(i)
		if index != -1:
			for ch in buf[:index][::-1]:
				if ch in Signs:
					bufL = ch + bufL
				else:
					break
			index = index + len(i)
			for ch in buf[index:]:
				if ch in Signs:
					bufR += ch
				else:
					break
			searchCond.append(bufL + i + bufR)
	
	def Check(buf):
		if buf.endswith(endBuf):
			endSearch = len(buf) - len(endBuf)
			startSearch = buf.rfind('\n', 0, len(buf) - len(endBuf))
			
			if startSearch == -1: startSearch = 0
			
			_buf = buf[startSearch:endSearch]
			for i in searchCond:
				if _buf.find(i) == -1: return False
			return True
		else:
			return False
	return Check		
		

class SSHError(Exception): pass

class TOCheck(object):
	"""
	Взято из: https://bitbucket.org/awgur/pylib/src/default/timetools.py	
	"""
	def __init__(self, timeOut):
		self.start = datetime.now()
		self.timeOut = timeOut
		                           
	def __call__(self, timeOut=None):
		if not timeOut:
			timeOut = self.timeOut
			
			buf = datetime.now() - self.start
			
            if buf.seconds > timeOut:
				return True
			else:
				return False
				
	def __bool__(self):
		return not self():


class SSHSubsh(object):
	"""Класс для облегчения работы с контекстами, когда приглашение коммандной оболочки 
	меняется в результате исполнения комманды.
	"""
	def __init__(self, ssh, prompt):
		self.ssh = ssh
		self.prompt = SSHPromptCheck(prompt)
		
	def __call__(self, cmd):
		res = self.ssh(cmd, wait=False)
		res += self.waitPrompt(self.prompt)
		return res

class SSHPromptCheck(object):
	"""
	Проверяет, появилось ли приглашение коммандной строки в выводе сервера
	
	Принимает:
	  - prompt:
	      Строка, которая должна однозначно идентифицировать приглашение 
	      коммандной строки, либо функция которой передаётся строка, и она 
	      должна вернуть True, если строка завершается приглашением коммандной
	      строки либо False в обратном случае. Кроме того можно передат сам объект,
	      созданный заранее.
	"""
	def __init__(self, prompt):
		if isinstance(prompt, self.__class__):
			self.prompt = prompt.prompt
			self.check = prompt.check
		else:
			def isFunc(): pass
			if type(prompt) == type(isFunc):
				self.check = prompt
				self.prompt = None
			else:
				self.prompt = prompt
			
			if self.prompt == None:
				raise SSHError('Prompt not set')
			
		
	def check(self, buf):
		if buf.endswith(self.prompt):
			return True
		else:
			return False
	
	def __call__(self, buf):
		if isinstance(buf, SSHOutput):
			return self.check(buf.out)
		elif buf:
			return self.check(buf)
		else:
			return False
			
		
class SSHOutput(object):
	"""Содержит вывод удалённого сервера и помогает управляться с ним.
	"""
	def __init__(self, out = '', err = ''):
		self.out = out
		self.err = err
		
	
	def error(self, buf):
		"Добавить буфер к буферу сообщения об ошибке"
		self.err += buf
		
	def __add__(self, buf):
		if isinstance(buf, self.__class__):
			self.out += buf.out
			self.err += buf.err
		else:	
			self.out += buf
		return self
		
	def __str__(self):
		return self.out
	
	def getError(self):
		return self.err
	
	def __bool__(self):
		if self.out or self.err:
			return True
		else:
			return False
	
	def __repr__(self):
		buf = ''
		buf += '---- OUTPUT -----------'
		buf += self.out
		buf += '---- ERRORS -----------'
		buf += self.err
		return buf
		
	def __getitem__(self, key):
		return self.out.splittines()[key]
		
	def __iter__(self):
		for line in self.out.splitlines():
			yield line
	
	def isEol(self):
		if self.out[-1] == '\n':
			return True
		else:
			return False
			
			
class SSHCommunicate(object):
	"""Взаимодействие с командой ssh. В сосотоянии отправлять ей данные и принимать данные от неё.
	
	Обеспчивает работу с exec_command в SSH
	"""
	def __init__(self, stdin, stdout, stderr):
		self.inC = stdin
		self.out = stdout
		self.err = stderr
		
	def __call__(self, buf=None):
		if buf != None:
			self.inC.write(buf)
		return SSHOutput(self.out.read(), self.err.read())
	
	def __str__(self):
		return str(self())
		
	def __repr__(self):
		return repr(self())

class _SSH(object):
	"""Метакласс для SSH соединений
	
	Принимает:
	  - host: Имя узла
	  - user: Имя пользователя
	  - passwd(не обязательно, если есть key): пароль пользователя
	  - key(не обязательно если есть passwd): файл ключа
	  - hostKeyPol[HOSTKEY_autoAdd | HOSTKEY_warn | HOSTKEY_reject]:
	      HOSTKEY_autoAdd(По умолчанию): 
	        если ключ удалённого узла нам неизвестен, добавить, соединение разрешить.
	      HOSTKEY_warn: если ключ удалённого узла нам неизвестен, предупредить.
	      HOSTKEY_reject: если ключ удалённого узла нам неизвестен, разорвать соединение.
	  - timeout: таймаут операций с сервером.
	"""
	def __init__(self, host, user, passwd=None, key=None, 
			hostKeyPol=HOSTKEY_autoAdd, timeout=None
		):
		if passwd == None and key == None:
			raise SSHError('Auth type not set')
				
		if hostKeyPol == HOSTKEY_autoAdd:
			_hostKeyPol = paramiko.AutoAddPolicy()
		elif hostKeyPol == HOSTKEY_warn:
			_hostKeyPol = paramiko.WarningPolicy()
		elif hostKeyPol == HOSTKEY_reject:
			_hostKeyPol = paramiko.RejectPolicy()
		else:
			raise SSHError('Unknown policy type')
		
		self.ssh = paramiko.SSHClient()	
		self.ssh.set_missing_host_key_policy(_hostKeyPol)	
		if passwd:
			self.ssh.connect(hostname=host, username=user, password=passwd, look_for_keys=False, allow_agent=False)
		else:
			self.ssh.connect(hostname=host, username=user, key_filename=key, look_for_keys=False, allow_agent=False)
			
		if timeout != None:
			self.timeout = timeout
		else:
			self.timeout = TIMEOUT
		
		self.user = user
		self.host = host
	
	def __del__(self):
		try:
			self.ssh.close()
		except:
			pass
		del self.ssh		
		
class InterSSH(_SSH):
	"""Класс Взаимодействия с сервером SSH
	
	Кроме аргументов необходимых для метакласса принимает:
	  - prompt: 
	      Параметр конструктора SSHPromptCheck. Принимается строго по имени.
	"""
	def __init__(self, *a, **kwa):
		if 'prompt' in kwa:
			prompt = kwa['prompt']
			del kwa['prompt']
			
		_SSH.__init__(self, *a, **kwa)
		
		self.shell = self.ssh.invoke_shell()
		self.shell.settimeout(self.timeout)
		if prompt == None:
			prompt = self._getPrompt()
			self.ready = True
		else:
			self.ready = False
		self.prompt = SSHPromptCheck(prompt)

		
	def __bool__(self):
		"Готово ли соединение принимать новые комманды"
		return self.ready
	
	def _getPrompt(self):
		"""Возвращает объект проверки на приглашение комммандной строки из текщего
		буфера вывода.
		"""
		buf = ''
		while self.shell.recv_ready():
			buf += self.shell.recv(1024)
		
		func = promptChecker(self.user, self.host, buf)
		return SSHPromptCheck(func)

	
	def waitPrompt(self, prompt=None):
		"""Ожидание приглашения коммандной строки
		
		Принимает необязательный аргумент: Экземпляр класса SSHPromptCheck, с помощью котороого 
		проверяется присутствие приглашения командного интерпретатора в выводе.
		"""
		if prompt == None:
			prompt = self.prompt
		else:
			prompt = SSHPromptCheck(prompt)
			
		if not self.ready:
			timeCheck = TOCheck(self.timeout)
			res = SSHOutput()
			while True:
				while self.shell.recv_stderr_ready():
					res.error(self.shell.recv_stderr(1024))

				while self.shell.recv_ready():
					res += self.shell.recv(1024)

				if prompt(res):
					break
		
				if timeCheck():
					raise SSHError('Command call timeout\n' + repr(res))
		
		self.ready = True
		return res

	def __call__(self, cmd, wait=True):
		"""Запуск команды
		
		Принимает:
		  - cmd: Комманда
		  - wait(необязательный аргумент): Признак ожидания завершения комманды.
		  
		Возвращает: Экземпляр класса SSHOutput
		"""
		res = SSHOutput()
		if not self.ready:
			res += self.waitPrompt()
		bytes = self.shell.send('%s\n' % cmd)
		if bytes == 0:
			raise SSHError('Channel closed')
		if wait:
			res += self.waitPrompt()
		else:
			self.ready = False
		return res

	def subShell(self, cmd, prompt=None):
		"""Возвращает объект SSHSubsh, через который удобно работать 
		интерпретаторами, запускаемыми из текущего, либо с ситуацией изменения приглашения 
		командного интерпретатора. 
		"""
		self(cmd, wait=False)
		if prompt == None:
			prompt = self._getPropt()
		return SSHSubsh(self, prompt)
			
	def close(self):
		res = SSHOutput
		if not self:
			res += self.waitPrompt()
		self.ssh.close()
		return res

class SSH(_SSH):
	"""Класс призванный управлять выполнением единичных комманд, без привлечения контекста.
	
	Принимает аргументы необходимые для метакласса _SSH
	"""
	def __init__(self, *a, **kwa):
		_SSH.__init__(self, *a, **kwa)
		
	def __call__(self, cmd):
		return SSHCommunicate(*self.ssh.exec_command(cmd, timeout=self.timeout))
		