# coding: utf-8
from configparser import ConfigParser
from dataclasses import is_dataclass, fields, Field, MISSING
from typing import Iterable, Optional, Dict, Any, List, Callable, Union
from threading import RLock
from os import getenv


class ConfigParseHelperError(Exception):
    """\
    Ошибки в модуле помощника разборщика конфигураций
    """


class NoSectionNotification(ConfigParseHelperError):
    """\
    Оповещение об отсутствующей секции конфигурации
    """


class TypeDescriber:
    """\
    Реализует паттерн "адаптер" поверх типов, для упрощения приведения значений типов к объявленным формам
    """
    def __init__(self, name, cast, like, is_complex=False):
        self.__name__ = name
        self.cast = cast
        self.like = like
        self.is_complex = is_complex

    def check(self, instance):
        if self.like is None:
            return False

        else:
            return check_instance(instance, self.like)

    def __repr__(self):
        return f'<TypeDescriber({self.__name__}, {self.like})>'

    def __call__(self, val):
        if val is None:
            return None

        elif not self.is_complex and check_instance(val, self.like):
            return val

        else:
            return self.cast(val)


def check_instance(obj, types):
    if types is None:
        return False

    elif isinstance(types, (tuple, list)):
        _flag = False
        for t in types:
            if check_instance(obj, t):
                _flag = True
                break

        return _flag

    elif isinstance(types, TypeDescriber):
        return types.check(obj)

    else:
        return isinstance(obj, types)


def cast_iterator(t: Union[type, TypeDescriber], lst: Iterable):
    """\
    Обрабатывает последовательности единого типа.
    """
    for i in lst:
        if check_instance(i, t):
            yield i

        else:
            try:
                yield t(i)

            except (TypeError, ValueError) as e:
                raise ValueError(f'Не удалось привести значение к нужному типу: '
                                 f'тип={t.__name__}; знач={i}')


def multi_item_tuple(tt, val):
    """\
    Обрабатывает кортежи, состоящие из нескольких значений типов (кортежи элементов разных типов)
    :param tt: Последовательность составляющих кортеж типов
    :param val: итерируемый объект, хранящий значения в указанном порядке.
    """
    val = list(val)
    t_len = len(tt)
    if t_len == 0:
        raise ValueError('При вызове процедуры конвертации котежей, не были указаны типы')

    if len(val) != t_len:
        raise ValueError(f'Значение не содержит положенных {t_len} элементов: {len(val)} - {val}')

    res = []

    for i in range(t_len):
        if check_instance(val[i], tt[i]):
            res.append(val[i])

        else:
            try:
                res.append(tt[i](val[i]))

            except (TypeError, ValueError) as e:
                raise ValueError(f'Не удалось привести значение к нужному типу: '
                                 f'тип={tt[i].__name__}; знач={val[i]}')

    return tuple(res)


def union_processor(tt, val):
    """\
    Пытается привести значение к одному из указанных типов
    :param tt: список возможных типов
    :param val: приводимое значение
    """
    if val is None:
        return val

    res = None
    ex = []

    if len(tt) == 0:
        raise ValueError('Не указан ни один тип в составном типе Union')

    sorted_types_begin = [ t for t in tt if t in (int, float, bool)]
    sorted_types_body = [ t for t in tt if t not in (int, float, bool, str)]
    sorted_types_end = [ t for t in tt if t == str]

    for t in sorted_types_begin + sorted_types_body + sorted_types_end:
        _t = get_type_describer(t)
        try:
            res = _t(val)
            break

        except (TypeError, ValueError) as e:
            ex.append(f'{t}: {e}')

    if res is None:
        raise ValueError('Не удалось привести значение не к одному из типов:\n' + '\n'.join(ex))

    else:
        return res


def dict_processor(tt, val):
    if len(tt) != 2:
        raise ValueError(f'Попытка воссоздать словарь со странным количеством аргументов типа: {tt}')

    try:
        _d = dict(val)

    except (TypeError, ValueError) as e:
        raise ValueError(f'Не удалось воссоздать словарь из представленного значения: {e}')

    _p = []

    for k, v in _d.items():
        try:
            _p.append((tt[0](k), tt[1](v)))

        except (TypeError, ValueError) as e:
            raise ValueError(f'Не удалось привести значения элемента словаря к требуемому типу: '
                             f'key="{k}" value="{v}"')

    return dict(_p)


def get_type_describer(t) -> TypeDescriber:
    if type(t).__name__ == '_GenericAlias':
        try:
            _args = t.__args__

        except AttributeError:
            raise TypeError(f'Неизвестный тип хранения внутренних типов для представителя сложного '
                            f'типа "_GenericAlias": {t}')

        if t.__name__ == 'List':
            try:
                _t = _args[0]

            except IndexError:
                raise ValueError(f'Тип {t} не содержит в себе типа своих элементов')

            return TypeDescriber(
                name=f'{t.__name__}[{_t.__name__}]',
                cast=lambda x: list(cast_iterator(get_type_describer(_t).cast, x)),
                like=list,
                is_complex=True
            )

        elif t.__name__ == 'Tuple':
            if not _args:
                raise ValueError(f'Тип {t} не содержит в себе пояснений по составу хранящихся в нём типов')

            if len(_args) == 1:
                _t = _args[0]

                return TypeDescriber(
                    name=f'Tuple[{_t.__name__}]',
                    cast=lambda x: list(cast_iterator(get_type_describer(_t).cast, x)),
                    like=tuple,
                    is_complex=True
                )

            else:
                _name = ', '.join(map(lambda x: x.__name__, _args))
                _cast_args = tuple(get_type_describer(i) for i in _args)

                return TypeDescriber(
                    name=f'Tuple[{_name}]',
                    cast=lambda x: multi_item_tuple(_cast_args, x),
                    like=tuple,
                    is_complex=True
                )

        elif t.__name__ == 'Dict':
            if len(_args) != 2:
                raise ValueError(f'Неожиданное количество значений типа в составном типе Dict: '
                                 f'{len(_args)} values=({_args})')

            _name = ', '.join(map(lambda x: x.__name__, _args))

            return TypeDescriber(
                name=f'Dict[{_name}]',
                cast=lambda x: dict_processor(_args, x),
                like=dict,
                is_complex=True
            )

        else:
            raise ValueError(f'Неизвестный представитель типа "_GenericAlias": {t.__name__}')

    elif type(t).__name__ == '_UnionGenericAlias':
        if t.__name__ not in ('Union', 'Optional'):
            raise TypeError(f'Неизвестный подтип _UnionGenericAlias: {t.__name__}')

        try:
            _args = t.__args__

        except AttributeError:
            raise TypeError(f'Неизвестный тип хранения внутренних типов для представителя сложного '
                            f'типа "_UnionGenericAlias": {t}')

        if len(_args) == 0:
            raise ValueError('Не указан ни один тип в конструкции Union')

        _cast_args = tuple(map(get_type_describer, _args))
        _args_name = ', '.join(map(lambda x: x.__name__, _args))

        return TypeDescriber(
            name=f'Union[{_args_name}]',
            cast=lambda x: union_processor(_cast_args, x),
            like=None
        )

    else:
        return TypeDescriber(
            name=t.__name__,
            cast=t,
            like=t
        )


class CPHSectionBase:
    """\
    Базовый класс обработки секции конфигурационного файла
    """

    def get(self, config_prop_name: str, dc_prop_name: str):
        """\
        Получить свойство из конфигурационного файла
        """
        raise NotImplemented()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        raise NotImplemented()


class CPHAbsentSection(CPHSectionBase):
    """\
    Класс создаваемый на отсутствующую секцию конфигурационного файла
    """
    def get(self, config_prop_name: str, dc_prop_name: str):
        raise NoSectionNotification()

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type == NoSectionNotification:
            return True


class CPHParamGetter(CPHSectionBase):
    def __init__(self, parser_helper_object):
        self.ph = parser_helper_object
        self.params = {}

    def _add_param(self, param_name: str, param_val: Any, parser: Optional[Callable[[Any], Any]] = None):
        """\
        Непосредственное добавление полученного параметра со всеми проверками.
        """

        if parser is not None and param_val is not None:
            param_val = parser(param_val)

        fld = self.ph.fields.get(param_name)
        if not isinstance(fld, Field):
            raise ConfigParseHelperError(f'В классе данных отсутствует свойство "{param_name}", '
                                         f'которое мы должны заполнить из параметра конфигурации: {fld}')

        if param_val is not None:
            type_desc = get_type_describer(fld.type)
            try:
                res = type_desc(param_val)

            except (ValueError, TypeError) as e:
                raise ConfigParseHelperError(f'При приведении параметра к '
                                             f'заданному типу произошла ошибка: '
                                             f'значение="{param_val}" ошибка="{e}"')

        else:
            if fld.default is not MISSING:
                res = fld.default

            elif fld.default_factory is not MISSING:
                res = fld.default_factory()

            else:
                raise ConfigParseHelperError('В конфигурации не заданна обязательная опция')

        self.params[param_name] = res

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None:
            self.ph.add_params(self.params)

    def get(self, config_prop_name: str, dc_prop_name: str, parser: Optional[Callable[[Any], Any]] = None):
        raise NoSectionNotification()


class CPHSection(CPHParamGetter):
    """\
    Класс производящий разбор конкретной секции конфигурации
    """

    def __init__(self, parser_helper_object, section: str):
        super().__init__(parser_helper_object)
        self.section_name = section
        self.section = parser_helper_object.conf_parser[section]

    def get(self, config_prop_name: str, dc_prop_name: str, parser: Optional[Callable[[Any], Any]] = None):
        """\
        :param config_prop_name: Имя опции в файле конфигурации
        :param dc_prop_name: Имя свойства в классе данных, хранящем конфигурацию
        :param parser: Исполнимый обработчик значения, перед его помещением в конфигурацию
        """
        try:
            self._add_param(dc_prop_name, self.section.get(config_prop_name), parser)

        except ConfigParseHelperError as e:
            raise ConfigParseHelperError(f'Ошибка при разборе параметра "{config_prop_name}" '
                                         f'в секции "{self.section_name}": {e}')


class CPHEnvParser(CPHParamGetter):
    """\
    Класс для разбора переменных окружения в том же ключе, что и файла конфигурации
    """

    def get(self, env_name: str, dc_prop_name: str, parser: Optional[Callable[[Any], Any]] = None):
        """\
        :param env_name: Имя переменной окружения, хранящей опцию
        :param dc_prop_name: Имя свойства в классе данных, хранящем конфигурацию
        :param parser: Исполнимый обработчик значения, перед его помещением в конфигурацию
        """

        try:
            self._add_param(dc_prop_name, getenv(env_name), parser)

        except ConfigParseHelperError as e:
            raise ConfigParseHelperError(f'Ошибка при получении значения из переменной окружения "{env_name}": {e}')


class CPHObjectsListGetter:
    """\
    Помощник для случаев, когда в наборе секций хранится однотипный набор объектов
    """
    def __init__(self, config_object_class: type, config_parser: ConfigParser, sections: Iterable[str]):
        self.sections = sections
        self.conf_parser = config_parser

        if not is_dataclass(config_object_class):
            raise ConfigParseHelperError(f'Представленный в качестве представления объекта конфигурации '
                                         f'класс не является классом данных: {config_object_class.__name__}')

        self.res_obj = config_object_class
        self.fields = dict((fld.name, fld) for fld in fields(config_object_class))
        self.obj_list = []
        self.ident_list = []

    def add_params(self, params: Dict[str, Any]):
        try:
            self.obj_list.append(self.res_obj(**params))

        except (ValueError, TypeError) as e:
            raise ConfigParseHelperError(f'Ошибка создания объекта объекта конфигурации, '
                                         f'списка объектов конфигурации: {e}')

    def get(self, config_prop_name: str, dc_prop_name: str, parser: Optional[Callable[[Any], Any]] = None):
        """\
        Подготавливаем список соответствия названий параметров в секции конкретным свойствам данного
        в помощник класса

        :param config_prop_name: Имя опции в файле конфигурации
        :param dc_prop_name: Имя свойства в классе данных, хранящем конфигурацию
        :param parser: Исполнимый обработчик значения, перед его помещением в конфигурацию
        """

        self.ident_list.append((config_prop_name, dc_prop_name, parser))

    def get_config_objects(self) -> List[object]:
        for section in self.sections:
            try:
                with CPHSection(self, section) as section_helper:
                    for conf_prop, dc_prop, parser in self.ident_list:
                        section_helper.get(conf_prop, dc_prop, parser)

            except ConfigParseHelperError as e:
                raise ConfigParseHelperError(f'Ошибка при разборе секции "{section}": {e}')

        res = self.obj_list[:]
        self.obj_list.clear()

        return res


class ConfigParseHelper:
    """\
    Помощник разбора конфигурации
    """
    def __init__(self, config_object_class: type, required_sections: Optional[Iterable[str]] = None):
        """\
        :param config_object_class: Dataclass, который мы подготовили как хранилище параметров конфигурации
        :param required_sections: Перечисление секций конфигурации, которые мы требуем, чтобы были в файле
        """

        if required_sections is not None:
            self.req_sections = set(required_sections)

        else:
            self.req_sections = set()

        if not is_dataclass(config_object_class):
            raise ConfigParseHelperError(f'Представленный в качестве объекта конфигурации класс не является '
                                         f'классом данных: {config_object_class.__name__}')

        self.res_obj = config_object_class
        self.fields = dict((fld.name, fld) for fld in fields(config_object_class))
        self.conf_parser: Optional[ConfigParser] = None
        self.config_params = {}
        self.config_params_lock = RLock()

    def add_params(self, params: Dict[str, Any]):
        self.config_params_lock.acquire()
        try:
            self.config_params.update(params)

        finally:
            self.config_params_lock.release()

    def section(self, section_name: str) -> CPHSectionBase:
        if self.conf_parser is None:
            raise ConfigParseHelperError(f'Прежде чем приступать к разбору файла конфигурации стоит его загрузить')

        if self.conf_parser.has_section(section_name):
            return CPHSection(self, section_name)

        else:
            return CPHAbsentSection()

    def load(self, filename: str):
        res = ConfigParser()
        try:
            res.read(filename)

        except (TypeError, IOError, OSError, ValueError) as e:
            raise ConfigParseHelperError(f'Не удалось загрузить файл конфигурации: файл="{filename}" '
                                         f'ошибка="{e}"')

        missing_sections = self.req_sections - set(res.sections())

        if missing_sections:
            missing_sections = ', '.join(missing_sections)
            raise ConfigParseHelperError(f'В конфигурационном файле отсутствуют секции: {missing_sections}')

        self.conf_parser = res

    def get_config(self):
        try:
            return self.res_obj(**self.config_params)

        except (ValueError, TypeError) as e:
            raise ConfigParseHelperError(f'Не удалось инициализировать объект конфигурации: {e}')

    def get_sections(self):
        return self.conf_parser.sections()
