# coding: utf-8

from dataclasses import fields, is_dataclass, asdict
from typing import Union, Dict, Any, Iterable


def _dict_has(obj: Dict[str, Any], key: str) -> bool:
    return key in obj


def _dict_get(obj: Dict[str, Any], key: str) -> Any:
    return obj[key]


def _obj_has(obj: object, key: str) -> bool:
    return hasattr(obj, key)


def _obj_get(obj: object, key: str) -> Any:
    return getattr(obj, key)


class TypeDescriber:
    """\
    Реализует паттерн "адаптер" поверх типов, для упрощения приведения значений типов к объявленным формам
    """
    def __init__(self, name, cast, like, is_complex=False):
        self.__name__ = name
        self.cast = cast
        self.like = like
        self.is_complex = is_complex

    def check(self, instance):
        if self.like is None:
            return False

        else:
            return check_instance(instance, self.like)

    def __repr__(self):
        return f'<TypeDescriber({self.__name__}, {self.like})>'

    def __call__(self, val):
        if val is None:
            return None

        elif not self.is_complex and check_instance(val, self.like):
            return val

        else:
            return self.cast(val)


def check_instance(obj, types):
    if types is None:
        return False

    elif isinstance(types, (tuple, list)):
        _flag = False
        for t in types:
            if check_instance(obj, t):
                _flag = True
                break

        return _flag

    elif isinstance(types, TypeDescriber):
        return types.check(obj)

    else:
        return isinstance(obj, types)


def cast_iterator(t: Union[type, TypeDescriber], lst: Iterable):
    """\
    Обрабатывает последовательности единого типа.
    """
    for i in lst:
        if check_instance(i, t):
            yield i

        else:
            try:
                yield t(i)

            except (TypeError, ValueError) as e:
                raise ValueError(f'Не удалось привести значение к нужному типу: '
                                 f'тип={t.__name__}; знач={i}')


def multi_item_tuple(tt, val):
    """\
    Обрабатывает кортежи, состоящие из нескольких значений типов (кортежи элементов разных типов)
    :param tt: Последовательность составляющих кортеж типов
    :param val: итерируемый объект, хранящий значения в указанном порядке.
    """
    val = list(val)
    t_len = len(tt)
    if t_len == 0:
        raise ValueError('При вызове процедуры конвертации котежей, не были указаны типы')

    if len(val) != t_len:
        raise ValueError(f'Значение не содержит положенных {t_len} элементов: {len(val)} - {val}')

    res = []

    for i in range(t_len):
        if check_instance(val[i], tt[i]):
            res.append(val[i])

        else:
            try:
                res.append(tt[i](val[i]))

            except (TypeError, ValueError) as e:
                raise ValueError(f'Не удалось привести значение к нужному типу: '
                                 f'тип={tt[i].__name__}; знач={val[i]}')

    return tuple(res)


def union_processor(tt, val):
    """\
    Пытается привести значение к одному из указанных типов
    :param tt: список возможных типов
    :param val: приводимое значение
    """
    if val is None:
        return val

    res = None
    ex = []

    if len(tt) == 0:
        raise ValueError('Не указан ни один тип в составном типе Union')

    sorted_types_begin = [ t for t in tt if t in (int, float, bool)]
    sorted_types_body = [ t for t in tt if t not in (int, float, bool, str)]
    sorted_types_end = [ t for t in tt if t == str]

    for t in sorted_types_begin + sorted_types_body + sorted_types_end:
        _t = get_type_describer(t)
        try:
            res = _t(val)
            break

        except (TypeError, ValueError) as e:
            ex.append(f'{t}: {e}')

    if res is None:
        raise ValueError('Не удалось привести значение не к одному из типов:\n' + '\n'.join(ex))

    else:
        return res


def dict_processor(tt, val):
    if len(tt) != 2:
        raise ValueError(f'Попытка воссоздать словарь со странным количеством аргументов типа: {tt}')

    try:
        _d = dict(val)

    except (TypeError, ValueError) as e:
        raise ValueError(f'Не удалось воссоздать словарь из представленного значения: {e}')

    _p = []

    tt_cast = [get_type_describer(t) for t in tt]

    for k, v in _d.items():
        try:
            _p.append((tt_cast[0](k), tt_cast[1](v)))

        except (TypeError, ValueError) as e:
            raise ValueError(f'Не удалось привести значения элемента словаря к требуемому типу: '
                             f'key="{k}" value="{v}"')

    return dict(_p)


def get_type_describer(t) -> TypeDescriber:
    if type(t).__name__ == '_GenericAlias':
        try:
            _args = t.__args__

        except AttributeError:
            raise TypeError(f'Неизвестный тип хранения внутренних типов для представителя сложного '
                            f'типа "_GenericAlias": {t}')

        if t.__name__ == 'List':
            try:
                _t = _args[0]

            except IndexError:
                raise ValueError(f'Тип {t} не содержит в себе типа своих элементов')

            return TypeDescriber(
                name=f'{t.__name__}[{_t.__name__}]',
                cast=lambda x: list(cast_iterator(get_type_describer(_t).cast, x)),
                like=list,
                is_complex=True
            )

        elif t.__name__ == 'Tuple':
            if not _args:
                raise ValueError(f'Тип {t} не содержит в себе пояснений по составу хранящихся в нём типов')

            if len(_args) == 1:
                _t = _args[0]

                return TypeDescriber(
                    name=f'Tuple[{_t.__name__}]',
                    cast=lambda x: list(cast_iterator(get_type_describer(_t).cast, x)),
                    like=tuple,
                    is_complex=True
                )

            else:
                _name = ', '.join(map(lambda x: x.__name__, _args))
                _cast_args = tuple(get_type_describer(i) for i in _args)

                return TypeDescriber(
                    name=f'Tuple[{_name}]',
                    cast=lambda x: multi_item_tuple(_cast_args, x),
                    like=tuple,
                    is_complex=True
                )

        elif t.__name__ == 'Dict':
            if len(_args) != 2:
                raise ValueError(f'Неожиданное количество значений типа в составном типе Dict: '
                                 f'{len(_args)} values=({_args})')

            _name = ', '.join(map(lambda x: x.__name__, _args))

            return TypeDescriber(
                name=f'Dict[{_name}]',
                cast=lambda x: dict_processor(_args, x),
                like=dict,
                is_complex=True
            )

        else:
            raise ValueError(f'Неизвестный представитель типа "_GenericAlias": {t.__name__}')

    elif type(t).__name__ == '_UnionGenericAlias':
        if t.__name__ not in ('Union', 'Optional'):
            raise TypeError(f'Неизвестный подтип _UnionGenericAlias: {t.__name__}')

        try:
            _args = t.__args__

        except AttributeError:
            raise TypeError(f'Неизвестный тип хранения внутренних типов для представителя сложного '
                            f'типа "_UnionGenericAlias": {t}')

        if len(_args) == 0:
            raise ValueError('Не указан ни один тип в конструкции Union')

        _cast_args = tuple(map(get_type_describer, [ i for i in _args if i is not None]))
        _args_name = ', '.join(map(lambda x: x.__name__, _args))

        return TypeDescriber(
            name=f'Union[{_args_name}]',
            cast=lambda x: union_processor(_cast_args, x),
            like=None
        )

    else:
        return TypeDescriber(
            name=t.__name__,
            cast=t,
            like=t
        )


def dataobj_extract(obj: Union[object, Dict[str, Any]], dataclass_type: type) -> object:
    """\
    Извлекает объект данных из предоставленного объекта, путём получения из него
    указанных в классе данных аттрибутов и поиска их в данном объекте.
    """

    params = {}

    if isinstance(obj, dict):
        _has = _dict_has
        _get = _dict_get

    else:
        _has = _obj_has
        _get = _obj_get

    if not is_dataclass(dataclass_type):
        raise ValueError(f'Не относится к классам данных: {dataclass_type.__name__}')

    for fld in fields(dataclass_type):
        if _has(obj, fld.name):
            val = _get(obj, fld.name)
            type_desc = get_type_describer(fld.type)
            if val is not None:
                try:
                    val = type_desc(val)

                except (ValueError, TypeError) as e:
                    raise ValueError(f'Аттрибут {fld.name} не может быть получен из значения "{val}"'
                                     f' с типом {type(val).__name__} поскольку не может быть преобразован в'
                                     f' тип {type_desc}, заданный в классе данных: {e}')

            params[fld.name] = val

    try:
        res = dataclass_type(**params)

    except (ValueError, TypeError) as e:
        _params = ', '.join(map(lambda x: f'{x[0]}="{x[1]}"', params.items()))
        raise ValueError(f'Не удалось получить объект'
                         f' класс {dataclass_type.__name__}'
                         f' из параметров: {_params}'
                         f' ошибка: {e}')

    return res


def json_type_sanitizer(val):
    """\
    Преобразует значение ``val`` в пригодное для преобразования в json значение.
    """

    val_t = type(val)

    if is_dataclass(val):
        return json_type_sanitizer(asdict(val))

    elif val_t in (int, float, str, bool) or val is None:
        return val

    elif val_t in (list, tuple):
        return list(map(json_type_sanitizer, val))

    elif val_t == dict:
        return dict((key, json_type_sanitizer(d_val)) for key, d_val in val.items())

    else:
        return str(val)
