#!/usr/bin/env python
# -*- coding: utf-8 -*-
""" Логирование в системный журнал Unix

Метки в журнале о уровне сообщения:
  "`": Debug
  ".": Info
  "*": Warning
  "!": Error
  "#": Alert
"""
import syslog
from sys import argv, version_info, exc_info
from time import time
from traceback import extract_tb

LOG_FACILITY = {
    'auth': syslog.LOG_AUTH,
    'authpriv': syslog.LOG_AUTH,
    'cron': syslog.LOG_CRON,
    'daemon': syslog.LOG_DAEMON,
    'ftp': syslog.LOG_DAEMON,
    'kern': syslog.LOG_KERN,
    'lpr': syslog.LOG_LPR,
    'mail': syslog.LOG_MAIL,
    'news': syslog.LOG_NEWS,
    'syslog': syslog.LOG_SYSLOG,
    'user': syslog.LOG_USER,
    'uucp': syslog.LOG_UUCP,
    'local0': syslog.LOG_LOCAL0,
    'local1': syslog.LOG_LOCAL1,
    'local2': syslog.LOG_LOCAL2,
    'local3': syslog.LOG_LOCAL3,
    'local4': syslog.LOG_LOCAL4,
    'local5': syslog.LOG_LOCAL5,
    'local6': syslog.LOG_LOCAL6,
    'local7': syslog.LOG_LOCAL7
}

# --- INTERFACE --- #
FACILITY = LOG_FACILITY['user']
def logPrep(ident, facility='user'):
    global FACILITY
    if not facility.lower() in LOG_FACILITY:
        raise LoggerError('Unknown facility')
    syslog.openlog(ident, syslog.LOG_PID)

    FACILITY = LOG_FACILITY[facility]


# --- ABSTRACT PART --- #

LOG_DEBUG = "`"
LOG_INFO = "."
LOG_WARN = "*"
LOG_ERR = "!"
LOG_ALERT = "#"

class LoggerError(Exception): pass

if version_info.major == 3:
    def _strWrap(msg):
        return str(msg)
elif version_info.major == 2:
    def _strWrap(msg):
        if isinstance(msg, unicode):
            return msg
        else:
            return str(msg).decode('utf-8')
else:
    raise LoggerError('Unknown major version of Python')


def _splitLines(obj):
    if isinstance(obj, (list, tuple)):
        for i in _parseList(obj):
            yield i
    elif isinstance(obj, dict):
        for i in _parseDict(obj):
            yield i
    else:
        buf = _strWrap(obj).splitlines()
        for i in buf:
            if len(i) < 401:
                yield i
            else:
                i = '<| ' + i + ' |>'
                lenI = len(i)
                c = 0
                while (c + 401) < lenI:
                    yield i[c:c+401]
                    c += 401
                yield i[c:]

def _parseList(lst):
    for i in lst:
        i = _splitLines(i)
        try:
            yield '- %s' % next(i)
            for l in i:
                yield '  %s' % l
        except StopIteration: pass

def _parseDict(dct):
    for key in sorted(dct):
        yield '%s:' % key
        for l in _splitLines(dct[key]):
            yield '    %s' % l

def _parseArgs(a, kwa):
    if a:
        yield '| --- ARGS:'
        for i in _parseList(a):
            yield '| %s' % i

    if kwa:
        yield '| --- NAMED ARGS:'
        for i in _parseDict(kwa):
            yield '| %s' % i

    if a or kwa:
        yield '| ---'

class TimingLog():
    def __init__(self, logproc, opName):
        self.logproc = logproc
        self.opName = opName
        self.dt = time()

    def __str__(self):
        return '%s :: %.4f' % (self.opName, (time() - self.dt))

    def __call__(self, msg, *a, **kwa):
        self.logproc('%s | %s' % (self, msg))
        for i in _parseArgs(a, kwa):
            self.logproc('%s | %s' % (self, i))

    def reset(self):
        self.dt = time()


class OperationContext(object):
    def __init__(self, log, opName):
        self.opName = opName
        self.log = log
        self.msgBuf = []

    def __call__(self, msg, *a, **kwa):
        self.msgBuf.append((msg, [ i for i in map(_strWrap, a)]))

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is not None:
            self.log.err("Operation '%s' fail" % self.opName)
            self.log.excpt_handler(exc_type, exc_val, exc_tb)

            if self.msgBuf:
                self.log.err('--- MESSAGES ---')
                for msg, a, kwa in self.msgBuf:
                    self.log.err(msg, a)
                self.log.err('--- END MESSAGES ---')


class AbstractLogger(object):
    def err(self, msg, *a, **kwa):
        raise LoggerError('NotImplemented')

    def __call__(self, msg, *a, **kwa):
        raise LoggerError('NotImplemented')

    def excpt_handler(self, eType, eObj, eTb):
        eTb = extract_tb(eTb)
        eType = str(eType)
        try:
            eType = eType.split("'")[1]
        except IndexError:
            pass

        try:
            eArgs = eObj.args[1:]
        except:
            eArgs = []

        try:
            eKwa = eObj.kwa
        except:
            eKwa = {}

        eObj = str(eObj)

        self.err('--- EXCEPTION ---')
        self.err(eType)

        for l in _splitLines(eObj):
            self.err('  ' + l)

        for l in _parseArgs(eArgs, eKwa):
            self.err(l)

        self.err('--- TRACEBACK ---')
        for _tbFile, _tbLine, _tbFunc, _tbText in eTb:
            self.err('File: %s, line %s in %s' % (_tbFile, _tbLine, _tbFunc))
            self.err('  %s' % _tbText)
        self.err('--- END EXCEPTION ---')

    def excpt(self, msg, *a, **kwa):
        eType, eObj, eTb = exc_info()
        self.err(msg, *a, **kwa)
        self.excpt_handler(eType, eObj, eTb)

    def cntxt(self, opName):
        """
        Если операцию нужно залогировать но при этом совершенно не хочется покрывать код
        толстым слоем try ... except можно завернуть операцию в контекст, и даже сохранить
        возвращаемом логером объекте отладочную информацию. При падении и только при нём
        это обязательно попадёт в лог.

        ОСТОРОЖНО ПАМЯТЬ!!!!!
        Объект копирует себе все переданные данные и опреобразует их сразу в строки
        Если объекту передать одни и те же параметры они всё равно будут храниться
        по копии на каждый вызов в виде строк, что может сьесть много памяти и вызвать
        болезненное её особождение.

        :param opName: членораздельное имя операции
        """
        return OperationContext(self, opName)

    def getTiming(self, opName):
        return TimingLog(self, opName)

    def dumpDict(self, msg, dct):
        self('--- ' + _strWrap(msg))
        for i in _parseDict(dct):
            self('| ' + i)
        self('---')
    
    @staticmethod
    def logPrep(*a, **kwa):
        logPrep(*a, **kwa)

# --- END ABSTRACT PART --- #

class Logger(AbstractLogger):
    facility = FACILITY
    def __init__(self, modName = 'main'):
        self.name = modName

    def _write(self, flags, prefix, msg):
        syslog.syslog(flags, u"%s %s: %s" % (prefix, self.name, msg))

    def __call__(self, msg, *a, **kwa):
        _flags = self.facility | syslog.LOG_INFO
        _prefix = LOG_INFO + ' ' + self.name + ': '
        syslog.syslog(_flags, _prefix + _strWrap(msg))
        for i in _parseArgs(a, kwa):
            syslog.syslog(_flags, _prefix + i)

    def warn(self, msg, *a, **kwa):
        _flags = self.facility | syslog.LOG_WARNING
        _prefix = LOG_WARN + ' ' + self.name + ': '
        syslog.syslog(_flags, _prefix + _strWrap(msg))
        for i in _parseArgs(a, kwa):
            syslog.syslog(_flags, _prefix + i)

    def err(self, msg, *a, **kwa):
        _flags = self.facility | syslog.LOG_ERR
        _prefix = LOG_ERR + ' ' + self.name + ': '
        syslog.syslog(_flags, _prefix + _strWrap(msg))
        for i in _parseArgs(a, kwa):
            syslog.syslog(_flags, _prefix + i)






