# coding: utf-8

from dataclasses import fields, dataclass, is_dataclass
from typing import Union, Dict, Any


def _dict_has(obj: Dict[str, Any], key: str) -> bool:
    return key in obj


def _dict_get(obj: Dict[str, Any], key: str) -> Any:
    return obj[key]


def _obj_has(obj: object, key: str) -> bool:
    return hasattr(obj, key)


def _obj_get(obj: object, key: str) -> Any:
    return getattr(obj, key)


def dataobj_extract(obj: Union[object, Dict[str, Any]], dataclass_type: type) -> dataclass:
    """\
    Извлекает объект данных из предоставленного объекта, путём получения из него
    указанных в классе данных аттрибутов и поиска их в данном объекте.
    """

    params = {}

    if isinstance(obj, dict):
        _has = _dict_has
        _get = _dict_get

    else:
        _has = _obj_has
        _get = _obj_get

    if not is_dataclass(dataclass_type):
        raise ValueError(f'Не относится к классам данных: {dataclass_type.__name__}')

    for fld in fields(dataclass_type):
        if _has(obj, fld.name):
            val = _get(obj, fld.name)
            if val is not None and not isinstance(val, fld.type):
                try:
                    val = fld.type(val)

                except (ValueError, TypeError) as e:
                    raise ValueError(f'Аттрибут {fld.name} не может быть получен из значения "{val}"'
                                     f' с типом {type(val).__name__} поскольку не может быть преобразован в'
                                     f' тип {fld.type.__name__}, заданный в классе данных: {e}')

            params[fld.name] = val

    try:
        res = dataclass_type(**params)

    except (ValueError, TypeError) as e:
        _params = ', '.join(map(lambda x: f'{x[0]}="{x[1]}"', params.items()))
        raise ValueError(f'Не удалось получить объект'
                         f' класс {dataclass_type.__name__}'
                         f' из параметров: {_params}'
                         f' ошибка: {e}')

    return res
