# coding: utf-8
"""
Тривиальная реализация сислоггера без блэк-джека и поэтесс

Метки в журнале о уровне сообщения:
  "`": Debug
  ".": Info
  "*": Warning
  "!": Error
  "#": Alert

"""
from time import time
from datetime import timedelta

import syslog
from traceback import extract_tb, extract_stack
from sys import exc_info


class Timing(object):
    def __init__(self, name=None):
        if name is None:
            self.prefix = ''
        else:
            self.prefix = '%s :: ' % name
        self.tsAll = time()
        self.ts = self.tsAll

    def getTime(self):
        return time() - self.ts

    def reset(self):
        self.ts = time()
        self.tsAll = self.ts

    def __str__(self):
        ts = time()
        return self.prefix + '%s(%.4f)' % (timedelta(seconds=(ts - self.tsAll)), ts - self.ts)

    def __call__(self, msg):
        _buf = '%s | %s' % (self, msg)
        self.ts = time()
        return _buf


class SimpleSysLogger(object):
    @staticmethod
    def initSyslog(ident):
        syslog.openlog(ident, syslog.LOG_PID)

    @staticmethod
    def getTiming(name=None):
        return Timing(name)

    def __init__(self, prefix, facility=syslog.LOG_USER):
        self.prefix = str(prefix)
        self.facility = facility

    def _write(self, flag, mark, msg):
        for l in str(msg).splitlines():
            syslog.syslog(self.facility | flag, '%s: %s %s' % (self.prefix, mark, l))

    def __call__(self, msg):
        self._write(syslog.LOG_INFO, '.', msg)
    
    def err(self, msg):
        self._write(syslog.LOG_ERR, '!', msg)
    
    def warn(self, msg):
        self._write(syslog.LOG_WARNING, '*', msg)
    
    def debug(self, msg):
        self._write(syslog.LOG_DEBUG, '`', msg)

    def alert(self, msg):
        self._write(syslog.LOG_ALERT, '#', msg)

    def sublog(self, prefix):
        return self.__class__('%s/%s' % (self.prefix, prefix), self.facility)

    def excpt(self, msg, eClass=None, eObj=None, eTb=None, stack_skip=0):
        if eClass is None:
            eClass, eObj, eTb = exc_info()
        
        tbData_tb = list(extract_tb(eTb))[::-1]
        tbData_stack = list(extract_stack())[::-1][(2 + stack_skip):]
        self.err(msg)
        self.err('--- EXCEPTION ---')
        self.err(' %s (%s)' % (eClass.__name__, eObj))
        self.err('--- TRACEBACK ---')
        for _tbFile, _tbLine, _tbFunc, _tbText in tbData_tb:
            self.err('File: %s, line %s in %s' % (_tbFile, _tbLine, _tbFunc))
            self.err('   %s' % _tbText)
        self.err('>>> Exception Handler <<<')
        for _tbFile, _tbLine, _tbFunc, _tbText in tbData_stack:
            self.err('File: %s, line %s in %s' % (_tbFile, _tbLine, _tbFunc))
            self.err('   %s' % _tbText)
        self.err('--- END EXCEPTION ---')
