# coding: utf-8
""" Логирование на консоль

Метки в журнале о уровне сообщения:
  "`": Debug
  ".": Info
  "*": Warning
  "!": Error
  "#": Alert
"""

from time import monotonic
from datetime import timedelta
from sys import exc_info, stderr, stdout
from traceback import extract_tb, extract_stack


class Timing(object):
    def __init__(self, name=None):
        if name is None:
            self.prefix = ''
        else:
            self.prefix = '%s :: ' % name
        self.tsAll = monotonic()
        self.ts = self.tsAll
    
    def getTime(self):
        return monotonic() - self.ts
    
    def reset(self):
        self.ts = monotonic()
        self.tsAll = self.ts
    
    def __str__(self):
        ts = monotonic()
        return self.prefix + '%s(%.4f)' % (timedelta(seconds=(ts - self.tsAll)), ts - self.ts)
    
    def __call__(self, msg):
        _buf = '%s | %s' % (self, msg)
        self.ts = monotonic()
        return _buf


class ConsoleLog(object):
    def __init__(self, appname='main'):
        self.appname = appname

    @staticmethod
    def _write(ItrCntnt):
        for l in ItrCntnt:
            print(l)

    def __call__(self, msg):
        self._write(map(
                lambda x: '%3s | %s :: %s' % ('.', self.appname, x),
                str(msg).splitlines()
        ))

    def err(self, msg):
        self._write(map(
            lambda x: '%3s | %s :: %s' % ('!', self.appname, x),
            str(msg).splitlines()
        ))

    def warn(self, msg):
        self._write(map(
            lambda x: '%3s | %s :: %s' % ('*', self.appname, x),
            str(msg).splitlines()
        ))

    def alert(self, msg):
        self._write(map(
            lambda x: '%3s | %s :: %s' % ('#', self.appname, x),
            str(msg).splitlines()
        ))

    def debug(self, msg):
        self._write(map(
            lambda x: '%3s | %s :: %s' % ('`', self.appname, x),
            str(msg).splitlines()
        ))

    @staticmethod
    def getTiming(name=None):
        return Timing(name)
    
    def sublog(self, name):
        return self.__class__('%s/%s' % (self.appname, name))
    
    def excpt(self, msg, eClass=None, eObj=None, eTb=None, stack_skip=0):
        if eClass is None:
            eClass, eObj, eTb = exc_info()
            
        tbData_tb = list(extract_tb(eTb))[::-1]
        tbData_stack = list(extract_stack())[::-1][(2 + stack_skip):]
        self.err(msg)
        self.err('--- EXCEPTION ---')
        self.err(' %s (%s)' % (eClass.__name__, eObj))
        self.err('--- TRACEBACK ---')
        for _tbFile, _tbLine, _tbFunc, _tbText in tbData_tb:
            self.err('File: %s, line %s in %s' % (_tbFile, _tbLine, _tbFunc))
            self.err('   %s' % _tbText)
        self.err('>>> Exception Handler <<<')
        for _tbFile, _tbLine, _tbFunc, _tbText in tbData_stack:
            self.err('File: %s, line %s in %s' % (_tbFile, _tbLine, _tbFunc))
            self.err('   %s' % _tbText)
        self.err('--- END EXCEPTION ---')


class StderrLog(ConsoleLog):
    @staticmethod
    def _write(msgItr):
        for l in msgItr:
            stderr.write('%s\n' % l)
        stderr.flush()


class StdoutLog(ConsoleLog):
    @staticmethod
    def _write(msgItr):
        for l in msgItr:
            stdout.write('%s\n' % l)
        stdout.flush()
