# coding: utf-8
"""\
Получение из конфигурационного файла параметров в виде класса данных.

НЕ РАБОТАЕТ БЕЗ ОСТАЛЬНОГО МОДУЛЯ type_utils
"""

from configparser import ConfigParser
from dataclasses import is_dataclass, fields, Field, MISSING
from typing import Iterable, Optional, Dict, Any, List, Callable
from threading import RLock
from os import getenv

from .type_descriptor import get_type_describer


class ConfigParseHelperError(Exception):
    """\
    Ошибки в модуле помощника разборщика конфигураций
    """


class NoSectionNotification(ConfigParseHelperError):
    """\
    Оповещение об отсутствующей секции конфигурации
    """


class CPHSectionBase:
    """\
    Базовый класс обработки секции конфигурационного файла
    """

    def get(self, config_prop_name: str, dc_prop_name: str):
        """\
        Получить свойство из конфигурационного файла
        """
        raise NotImplemented()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        raise NotImplemented()


class CPHAbsentSection(CPHSectionBase):
    """\
    Класс создаваемый на отсутствующую секцию конфигурационного файла
    """
    def get(self, config_prop_name: str, dc_prop_name: str):
        raise NoSectionNotification()

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type == NoSectionNotification:
            return True


class CPHParamGetter(CPHSectionBase):
    def __init__(self, parser_helper_object):
        self.ph = parser_helper_object
        self.params = {}

    def _add_param(self, param_name: str, param_val: Any, parser: Optional[Callable[[Any], Any]] = None):
        """\
        Непосредственное добавление полученного параметра со всеми проверками.
        """

        if parser is not None and param_val is not None:
            param_val = parser(param_val)

        fld = self.ph.fields.get(param_name)
        if not isinstance(fld, Field):
            raise ConfigParseHelperError(f'В классе данных отсутствует свойство "{param_name}", '
                                         f'которое мы должны заполнить из параметра конфигурации: {fld}')

        if param_val is not None:
            type_desc = get_type_describer(fld.type)
            try:
                res = type_desc(param_val)

            except (ValueError, TypeError) as e:
                raise ConfigParseHelperError(f'При приведении параметра к '
                                             f'заданному типу произошла ошибка: '
                                             f'значение="{param_val}" ошибка="{e}"')

        else:
            if fld.default is not MISSING:
                res = fld.default

            elif fld.default_factory is not MISSING:
                res = fld.default_factory()

            else:
                raise ConfigParseHelperError('В конфигурации не заданна обязательная опция')

        self.params[param_name] = res

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None:
            self.ph.add_params(self.params)

    def get(self, config_prop_name: str, dc_prop_name: str, parser: Optional[Callable[[Any], Any]] = None):
        raise NoSectionNotification()


class CPHSection(CPHParamGetter):
    """\
    Класс производящий разбор конкретной секции конфигурации
    """

    def __init__(self, parser_helper_object, section: str):
        super().__init__(parser_helper_object)
        self.section_name = section
        self.section = parser_helper_object.conf_parser[section]

    def get(self, config_prop_name: str, dc_prop_name: str, parser: Optional[Callable[[Any], Any]] = None):
        """\
        :param config_prop_name: Имя опции в файле конфигурации
        :param dc_prop_name: Имя свойства в классе данных, хранящем конфигурацию
        :param parser: Исполнимый обработчик значения, перед его помещением в конфигурацию
        """
        try:
            self._add_param(dc_prop_name, self.section.get(config_prop_name), parser)

        except ConfigParseHelperError as e:
            raise ConfigParseHelperError(f'Ошибка при разборе параметра "{config_prop_name}" '
                                         f'в секции "{self.section_name}": {e}')


class CPHEnvParser(CPHParamGetter):
    """\
    Класс для разбора переменных окружения в том же ключе, что и файла конфигурации
    """

    def get(self, env_name: str, dc_prop_name: str, parser: Optional[Callable[[Any], Any]] = None):
        """\
        :param env_name: Имя переменной окружения, хранящей опцию
        :param dc_prop_name: Имя свойства в классе данных, хранящем конфигурацию
        :param parser: Исполнимый обработчик значения, перед его помещением в конфигурацию
        """

        try:
            self._add_param(dc_prop_name, getenv(env_name), parser)

        except ConfigParseHelperError as e:
            raise ConfigParseHelperError(f'Ошибка при получении значения из переменной окружения "{env_name}": {e}')


class CPHObjectsListGetter:
    """\
    Помощник для случаев, когда в наборе секций хранится однотипный набор объектов
    """
    def __init__(self, config_object_class: type, config_parser: ConfigParser, sections: Iterable[str]):
        self.sections = sections
        self.conf_parser = config_parser

        if not is_dataclass(config_object_class):
            raise ConfigParseHelperError(f'Представленный в качестве представления объекта конфигурации '
                                         f'класс не является классом данных: {config_object_class.__name__}')

        self.res_obj = config_object_class
        self.fields = dict((fld.name, fld) for fld in fields(config_object_class))
        self.obj_list = []
        self.ident_list = []

    def add_params(self, params: Dict[str, Any]):
        try:
            self.obj_list.append(self.res_obj(**params))

        except (ValueError, TypeError) as e:
            raise ConfigParseHelperError(f'Ошибка создания объекта объекта конфигурации, '
                                         f'списка объектов конфигурации: {e}')

    def get(self, config_prop_name: str, dc_prop_name: str, parser: Optional[Callable[[Any], Any]] = None):
        """\
        Подготавливаем список соответствия названий параметров в секции конкретным свойствам данного
        в помощник класса

        :param config_prop_name: Имя опции в файле конфигурации
        :param dc_prop_name: Имя свойства в классе данных, хранящем конфигурацию
        :param parser: Исполнимый обработчик значения, перед его помещением в конфигурацию
        """

        self.ident_list.append((config_prop_name, dc_prop_name, parser))

    def get_config_objects(self) -> List[object]:
        for section in self.sections:
            try:
                with CPHSection(self, section) as section_helper:
                    for conf_prop, dc_prop, parser in self.ident_list:
                        section_helper.get(conf_prop, dc_prop, parser)

            except ConfigParseHelperError as e:
                raise ConfigParseHelperError(f'Ошибка при разборе секции "{section}": {e}')

        res = self.obj_list[:]
        self.obj_list.clear()

        return res


class ConfigParseHelper:
    """\
    Помощник разбора конфигурации
    """
    def __init__(self, config_object_class: type, required_sections: Optional[Iterable[str]] = None):
        """\
        :param config_object_class: Dataclass, который мы подготовили как хранилище параметров конфигурации
        :param required_sections: Перечисление секций конфигурации, которые мы требуем, чтобы были в файле
        """

        if required_sections is not None:
            self.req_sections = set(required_sections)

        else:
            self.req_sections = set()

        if not is_dataclass(config_object_class):
            raise ConfigParseHelperError(f'Представленный в качестве объекта конфигурации класс не является '
                                         f'классом данных: {config_object_class.__name__}')

        self.res_obj = config_object_class
        self.fields = dict((fld.name, fld) for fld in fields(config_object_class))
        self.conf_parser: Optional[ConfigParser] = None
        self.config_params = {}
        self.config_params_lock = RLock()

    def add_params(self, params: Dict[str, Any]):
        self.config_params_lock.acquire()
        try:
            self.config_params.update(params)

        finally:
            self.config_params_lock.release()

    def section(self, section_name: str) -> CPHSectionBase:
        if self.conf_parser is None:
            raise ConfigParseHelperError(f'Прежде чем приступать к разбору файла конфигурации стоит его загрузить')

        if self.conf_parser.has_section(section_name):
            return CPHSection(self, section_name)

        else:
            return CPHAbsentSection()

    def load(self, filename: str):
        res = ConfigParser()
        try:
            res.read(filename)

        except (TypeError, IOError, OSError, ValueError) as e:
            raise ConfigParseHelperError(f'Не удалось загрузить файл конфигурации: файл="{filename}" '
                                         f'ошибка="{e}"')

        missing_sections = self.req_sections - set(res.sections())

        if missing_sections:
            missing_sections = ', '.join(missing_sections)
            raise ConfigParseHelperError(f'В конфигурационном файле отсутствуют секции: {missing_sections}')

        self.conf_parser = res

    def get_config(self):
        try:
            return self.res_obj(**self.config_params)

        except (ValueError, TypeError) as e:
            raise ConfigParseHelperError(f'Не удалось инициализировать объект конфигурации: {e}')

    def get_sections(self):
        return self.conf_parser.sections()
