# coding: utf-8

class TypeDescriptorInterface:
    """\
    Определяем общий интерфейс для описателей типов
    """
    def __init__(self, name, type_class, is_nullable=False):
        self.__name__ = name
        self.t_class = type_class
        self.is_nullable = is_nullable
        self.event_description = ''
        self.union_sort_order = 3

    def raise_nullable(self, msg=''):
        msg = msg if not msg else f': {msg}'

        if not self.is_nullable:
            raise ValueError(f'Передача значения "None" переменной, которая этого не ожидает{msg}')

    def __call__(self, val):
        """\
        Реализация приведения типа
        """
        raise NotImplemented()

    def __repr__(self):
        if self.t_class is None:
            return self.__name__

        else:
            return f'{self.__name__}({self.t_class.__name__})'

    def check(self, val):
        """\
        Реализация проверки соответствия типу
        """
        raise NotImplemented()


class BaseTypeDescriptor(TypeDescriptorInterface):
    """\
    Базовый класс, объявляющий общий интерфейс классов адаптеров проверки и преобразования типов
    """
    def check(self, val):
        event_description = ''
        res = False
        if val is None:
            event_description = 'Передано значение "None"'
            res = self.is_nullable

        elif isinstance(self.t_class, TypeDescriptorInterface):
            res = self.t_class.check(val)
            event_description = self.t_class.event_description

        else:
            event_description = f'Требуется тип "{self.t_class.__name__}" а получаем "{type(val).__name__}"'
            res = isinstance(val, self.t_class)

        if not res:
            self.event_description = event_description

        return res

    def check_fail(self, msg):
        self.event_description = msg
        return False


class ScalarTypeDescriptor(BaseTypeDescriptor):
    """\
    Реализация адаптера над простыми типами
    """

    def __init__(self, type_class, is_nullable=False):
        super().__init__(type_class.__name__, type_class, is_nullable)
        if type_class in (int, float, bool, complex):
            self.union_sort_order = 0

        elif type_class == str:
            self.union_sort_order = 2

        else:
            self.union_sort_order = 1

    def __call__(self, val):
        if val is None:
            return self.raise_nullable()

        if self.check(val):
            return val

        if self.union_sort_order == 1:
            if isinstance(val, (list, tuple)):
                return self.t_class(*val)

            elif isinstance(val, dict):
                return self.t_class(**val)

        return self.t_class(val)


class IterableTypeDescriptor(BaseTypeDescriptor):
    """\
    Реализация адаптера над последовательностями
    """
    def __init__(self, iterator_class, value_class, is_nullable=False):
        super().__init__(
            name=f'{iterator_class.__name__}[{value_class.__name__}]',
            type_class=value_class,
            is_nullable=is_nullable
        )

        self.iterator_class = iterator_class

    def check(self, vals):
        idx = 0
        for val in vals:
            if not super().check(val):
                return self.check_fail(f'Элемент {idx}: {self.event_description}')

            else:
                idx += 1

        return True

    def __call__(self, vals):
        if vals is None:
            return self.raise_nullable()

        res = []
        for val in vals:
            if val is None:
                self.raise_nullable(f'Элемент "{len(res)}"')

            try:
                res.append(self.t_class(val))

            except (TypeError, ValueError) as e:
                raise ValueError(f'Не удалось преобразовать элемент "{len(res)}" со значением "{val}" '
                                 f'в результирующий тип: {e}')

        return self.iterator_class(res)


class TupleTypeDescriptor(BaseTypeDescriptor):
    """\
    Адаптер над кортежами
    """
    def __init__(self, type_classes, is_nullable=False):
        if not isinstance(type_classes, (list, tuple)) or not type_classes:
            raise TypeError(f'В конструктор прокси для обработки кортежей не передано типов кортежа')

        names = ', '.join([i.__name__ for i in type_classes])

        super().__init__(f'Tuple[{names}]', None, is_nullable)

        self.type_classes = tuple(map(
            lambda x: x if isinstance(x, TypeDescriptorInterface) else ScalarTypeDescriptor(x),
            type_classes
        ))

    def check(self, vals):
        if not isinstance(vals, tuple):
            return self.check_fail(f'Переданная переменная не является кортежем, а относится к типу: '
                                   f'{type(vals).__name__}')

        if len(vals) != len(self.type_classes):
            return self.check_fail(f'Не достаточно элементов в кортеже: '
                                   f'имеется={len(vals)} нужно={len(self.type_classes)}')

        for i in range(len(self.type_classes)):
            if not self.type_classes[i].check(vals[i]):
                return self.check_fail(f'Элемент {i}, значение "{vals[i]}": '
                                       f'{self.type_classes[i].event_description}')

    def __call__(self, vals):
        if vals is None:
            return self.raise_nullable()

        if not isinstance(vals, tuple):
            raise ValueError(f'Переданная переменная не является кортежем, а относится к типу: '
                             f'{type(vals).__name__}')

        if len(vals) != len(self.type_classes):
            raise ValueError(f'Не достаточно элементов в кортеже: '
                             f'имеется={len(vals)} нужно={len(self.type_classes)}')

        res = []
        for i in range(len(self.type_classes)):
            try:
                res.append(self.type_classes[i](vals[i]))

            except (ValueError, TypeError) as e:
                raise ValueError(f'Не удалось привести к результирующему виду '
                                 f'элемент {i} со значением "{vals[i]}": {e}')

        return tuple(res)


class UnionTypeDescriptor(BaseTypeDescriptor):
    """\
    Адаптер для объединения типов
    """
    def __init__(self, type_classes, is_nullable=False):
        if not isinstance(type_classes, (list, tuple)) or not type_classes:
            raise TypeError(f'В конструктор прокси для обработки объединений типов не передано типов')

        names = ', '.join([i.__name__ for i in type_classes])

        super().__init__(f'Union[{names}]', None, is_nullable)

        self.type_classes = tuple(sorted(map(
            lambda x: x if isinstance(x, BaseTypeDescriptor) else ScalarTypeDescriptor(x),
            type_classes
        ), key=lambda x: x.union_sort_order))

    def check(self, val):
        for t in self.type_classes:
            if t.check(val):
                return True

        self.event_description = f'Значение "{val}" типа "{type(val).__name__}" не соответствует ' \
                                 f'ни одному типу из моего набора'
        return False

    def __call__(self, val):
        if val is None:
            return self.raise_nullable()

        if self.check(val):
            return val

        errs = []

        for t in self.type_classes:
            try:
                return t(val)

            except (TypeError, ValueError) as e:
                errs.append(f'{repr(t)}: {e}')

        raise ValueError(f'Не удалось преобразовать значение "{val}" типа "{type(val).__name__}" '
                         f'ни к одному из имеющихся типов: ' + '\n'.join(errs))


class DictTypeDescriptor(BaseTypeDescriptor):
    """\
    Адаптер словарей
    """
    def __init__(self, key_class, value_class, is_nullable=False):
        super().__init__(f'Dict[{key_class.__name__}, {value_class.__name__}]', None, is_nullable)

        if isinstance(key_class, BaseTypeDescriptor):
            self.key_class = key_class

        else:
            self.key_class = ScalarTypeDescriptor(key_class)

        if isinstance(value_class, BaseTypeDescriptor):
            self.value_class = value_class

        else:
            self.value_class = ScalarTypeDescriptor(value_class)

    def check(self, val):
        try:
            d = dict(val)

        except (TypeError, ValueError) as e:
            self.event_description = f'Не удалось преобразовать переданное значение в словарь: {e}'
            return False

        for k, v in d.items():
            if not self.key_class.check(k):
                return self.check_fail(f'В паре ["{k}": "{v}"] ключ не соответствует ожидаемому типу: ' 
                                       f'{self.key_class.event_description}')

            if not self.value_class.check(v):
                return self.check_fail(f'В паре ["{k}": "{v}"] значение не соответствует ожидаемому типу: '
                                       f'{self.value_class.event_description}')

        return True

    def __call__(self, val):
        try:
            d = dict(val)

        except (TypeError, ValueError) as e:
            raise ValueError(f'Не удалось преобразовать переданное значение в словарь: {e}')

        p = []
        for k, v in d.items():
            p.append((self.key_class(k), self.value_class(v)))

        return dict(p)


def get_type_describer(t) -> BaseTypeDescriptor:
    if type(t).__name__ == '_GenericAlias':
        try:
            _args = t.__args__

        except AttributeError:
            raise TypeError(f'Неизвестный тип хранения внутренних типов для представителя сложного '
                            f'типа "_GenericAlias": {t}')

        if t.__name__ == 'List':
            try:
                _t = _args[0]

            except IndexError:
                raise ValueError(f'Тип {t} не содержит в себе типа своих элементов')

            return IterableTypeDescriptor(
                iterator_class=list,
                value_class=get_type_describer(_t)
            )

        elif t.__name__ == 'Tuple':
            if not _args:
                raise ValueError(f'Тип {t} не содержит в себе пояснений по составу хранящихся в нём типов')

            if len(_args) == 1:
                _t = _args[0]

                return IterableTypeDescriptor(
                    iterator_class=tuple,
                    value_class=get_type_describer(_t)
                )

            else:
                return TupleTypeDescriptor(
                    type_classes=_args
                )

        elif t.__name__ == 'Dict':
            if len(_args) != 2:
                raise ValueError(f'Неожиданное количество значений типа в составном типе Dict: '
                                 f'{len(_args)} values=({_args})')

            return DictTypeDescriptor(
                key_class=get_type_describer(_args[0]),
                value_class=get_type_describer(_args[1])
            )

        else:
            raise ValueError(f'Неизвестный представитель типа "_GenericAlias": {t.__name__}')

    elif type(t).__name__ == '_UnionGenericAlias':
        if t.__name__ not in ('Union', 'Optional'):
            raise TypeError(f'Неизвестный подтип _UnionGenericAlias: {t.__name__}')

        nullable = False
        if t.__name__ == 'Optional':
            nullable = True

        try:
            _args = t.__args__

        except AttributeError:
            raise TypeError(f'Неизвестный тип хранения внутренних типов для представителя сложного '
                            f'типа "_UnionGenericAlias": {t}')

        if len(_args) == 0:
            raise ValueError('Не указан ни один тип в конструкции Union')

        type_classes = tuple(map(lambda x: get_type_describer(x), [ i for i in _args if i is not None]))

        return UnionTypeDescriptor(
            type_classes=type_classes,
            is_nullable=nullable
        )

    else:
        return ScalarTypeDescriptor(type_class=t)
