package main

import (
	"errors"
	"fmt"
	"log/syslog"
	"net"
	"os"
	"strings"
	"time"
)

const (
	LOG_IDENT string = "net-a0fs-gostun"
)

const (
	LOGP_INFO = iota
	LOGP_WARN
	LOGP_ERR
)

type logMessage struct {
	prio uint
	msg  string
}

var log_chan = make(chan logMessage, 100)

func log_handler() {
	var err error

	log, err := syslog.New(syslog.LOG_USER, LOG_IDENT)
	if err != nil {
		panic(err)
	}

	for log_msg := range log_chan {
		switch log_msg.prio {
		case LOGP_INFO:
			err = log.Info(log_msg.msg)
		case LOGP_WARN:
			err = log.Warning(log_msg.msg)
		case LOGP_ERR:
			err = log.Err(log_msg.msg)
		default:
			err = log.Err(fmt.Sprintf("Unknown type of log priority: %v", log_msg.prio))
		}

		if err != nil {
			panic(err)
		}
	}
}

func log_info(msg string, vars ...interface{}) {
	log_chan <- logMessage{
		prio: LOGP_INFO,
		msg:  fmt.Sprintf(msg, vars...)}
}

func log_warn(msg string, vars ...interface{}) {
	log_chan <- logMessage{
		prio: LOGP_WARN,
		msg:  fmt.Sprintf(msg, vars...)}
}

func log_err(msg string, vars ...interface{}) {
	log_chan <- logMessage{
		prio: LOGP_ERR,
		msg:  fmt.Sprintf(msg, vars...)}
}

func getPort() (res string, err error) {
	if len(os.Args) < 2 {
		err = errors.New("No port specify")
	} else {
		res = fmt.Sprintf(":%s", os.Args[1])
	}
	return
}

func fail(code int, msg string) {
	usage := "Usage: cmd port"
	usage += "\n  port: port number for listening clients"

	fmt.Fprintln(os.Stderr, msg)
	fmt.Fprintln(os.Stderr, usage)
	os.Exit(code)
	return
}

func handler(conn net.Conn) {
	_s_tm := time.Now()
	remoteAddr := conn.RemoteAddr().String()

	defer func() {
		var err error
		_s_tmd := time.Now().Sub(_s_tm)
		if r_err := recover(); r_err != nil {
			log_err("Error on service for %s (service time: %v): %v", remoteAddr, _s_tmd, r_err)
		} else {
			err = conn.Close()
			if err != nil {
				log_err("Error on closing socket when service %s (service time: %v): %v",
					remoteAddr, _s_tmd, err)

			} else {
				log_info("Ok service %s in %v", remoteAddr, _s_tmd)
			}
		}
	}()

	addr_spit := strings.Split(remoteAddr, ":")
	addr_last := len(addr_spit) - 1
	addr := strings.Join(addr_spit[:addr_last], ":")

	var dns string
	dns_names, err := net.LookupAddr(strings.Trim(addr, "[] "))
	if err != nil {
		log_warn("Fail to resolve dns name for %s: %v", remoteAddr, err)
	} else {
		for _, dns_name := range dns_names {
			dns += "Name: " + dns_name + "\n"
		}
	}
	addr = "IP: " + addr + "\n"
	port := "Port: " + addr_spit[addr_last] + "\n"

	if _, err := conn.Write([]byte(addr)); err != nil {
		panic(err.Error())
	}
	if _, err := conn.Write([]byte(dns)); err != nil {
		panic(err.Error())
	}
	if _, err := conn.Write([]byte(port)); err != nil {
		panic(err.Error())
	}
}

func main() {
	port, err := getPort()
	if err != nil {
		fail(1, fmt.Sprint("Error in parsing port number: ", err))
	}

	go log_handler()

	svc, err := net.Listen("tcp", port)
	if err != nil {
		fail(2, fmt.Sprint("Fail to open socket: ", err))
	}

	for {
		conn, err := svc.Accept()
		if err != nil {
			fail(1, fmt.Sprint("Error accepting connections: ", err))
		}
		go handler(conn)
	}
}
