package main

import (
	"errors"
	"fmt"
	"log/syslog"
	"net"
	"os"
	"strings"
	"time"
)

const (
	LOG_IDENT string = "net-a0fs-gostun"
)

func getPort() (res string, err error) {
	if len(os.Args) < 2 {
		err = errors.New("No port specify")
	} else {
		res = fmt.Sprintf(":%s", os.Args[1])
	}
	return
}

func fail(code int, msg string) {
	usage := "Usage: cmd port"
	usage += "\n  port: port number for listening clients"

	fmt.Fprintln(os.Stderr, msg)
	fmt.Fprintln(os.Stderr, usage)
	os.Exit(code)
	return
}

func handler(conn net.Conn) {
	log, err := syslog.New(syslog.LOG_USER, LOG_IDENT)
	if err != nil {
		panic(err)
	}

	_s_tm := time.Now()
	remoteAddr := conn.RemoteAddr().String()

	defer func() {
		var err error
		_s_tmd := time.Now().Sub(_s_tm)
		if r_err := recover(); r_err != nil {
			err = log.Err(fmt.Sprintf("Error on service for %s (service time: %v): %v", remoteAddr, _s_tmd, r_err))
			if err != nil {
				panic(err.Error())
			}
		} else {
			err = conn.Close()
			if err != nil {
				err = log.Err(fmt.Sprintf("Error on closing socket when service %s (service time: %v): %v",
					remoteAddr, _s_tmd, err))

				if err != nil {
					panic(err.Error())
				}
			} else {
				_ = log.Info(fmt.Sprintf("Ok service %s in %v", remoteAddr, _s_tmd))
			}
		}
	}()

	addr_spit := strings.Split(remoteAddr, ":")
	addr_last := len(addr_spit) - 1
	addr := "IP: " + strings.Join(addr_spit[:addr_last], ":") + "\n"
	port := "Port: " + addr_spit[addr_last] + "\n"

	if _, err := conn.Write([]byte(addr)); err != nil {
		panic(err.Error())
	}
	if _, err := conn.Write([]byte(port)); err != nil {
		panic(err.Error())
	}
}

func main() {
	port, err := getPort()
	if err != nil {
		fail(1, fmt.Sprint("Error in parsing port number: ", err))
	}

	svc, err := net.Listen("tcp", port)
	if err != nil {
		fail(2, fmt.Sprint("Fail to open socket: ", err))
	}

	for {
		conn, err := svc.Accept()
		if err != nil {
			fail(1, fmt.Sprint("Error accepting connections: ", err))
		}
		go handler(conn)
	}
}
