# coding: utf-8
from configparser import ConfigParser, SectionProxy
from sys import argv
from os.path import isdir
from typing import Optional, Iterable


from .error import Error


class ConfigError(Error):
    pass


COMMON_SECTION = 'common'
MAIN = 'main'
SMTP = 'smtp'
DEFAULT_PG_PORT = 5432


class DBTask(object):
    def __init__(self, task_name: str, host_name: str, db_name: str, user_name: str, passwd: Optional[str],
                 dst_dir: str, port: int = DEFAULT_PG_PORT):
        self.host_name = host_name
        self.db_name = db_name
        self.user_name = user_name
        self.passwd = passwd
        self.dst_dir = dst_dir
        self.name = task_name
        self.port = port

        if not (self.db_name or self.user_name or self.dst_dir or self.host_name):
            raise ConfigError(f'Some important config parameters not set: '
                              f'db_name="{db_name}" '
                              f'user_name="{user_name}" '
                              f'dst_dir="{dst_dir}" '
                              f'host_name="{host_name}"')


class CommonTaskDescription(object):
    def __init__(self, host_name: Optional[str], user_name: Optional[str],
                 passwd: Optional[str], dst_dir: Optional[str], port: int = DEFAULT_PG_PORT):
        self.user_name = user_name
        self.passwd = passwd
        self.dst_dir = dst_dir
        self.host_name = host_name
        self.port = port

    def parse_section(self, config_section: SectionProxy) -> DBTask:
        db_name = config_section.get('db_name')
        user_name = config_section.get('user_name', self.user_name)
        passwd = config_section.get('passwd', self.passwd)
        dst_dir = config_section.get('dst_dir', self.dst_dir)
        host_name = config_section.get('host_name', self.host_name)
        port = config_section.getint('port', self.port)

        if not isdir(dst_dir):
            raise ConfigError(f'Destionation directory not exists for "{config_section.name}": {dst_dir}')

        return DBTask(task_name=config_section.name, host_name=host_name, db_name=db_name,
                      user_name=user_name, passwd=passwd, port=port, dst_dir=dst_dir)

    @classmethod
    def parse_config(cls, config: ConfigParser) -> Iterable[DBTask]:
        if not config.has_section(COMMON_SECTION):
            tmpl = cls(host_name=None, user_name=None, passwd=None, dst_dir=None)

        else:
            _section = config['common']
            host_name = _section.get('host_name', '127.0.0.1')
            user_name = _section.get('user_name')
            passwd = _section.get('passwd')
            dst_dir = _section.get('dst_dir')
            port = _section.getint('port', DEFAULT_PG_PORT)

            tmpl = cls(host_name=host_name, user_name=user_name, passwd=passwd, dst_dir=dst_dir, port=port)

        for _section in filter(lambda x: x not in (COMMON_SECTION, MAIN, SMTP), config.sections()):
            yield tmpl.parse_section(config[_section])


class Config(object):
    def __init__(self):
        try:
            config_file = argv[1]
        except IndexError:
            raise ConfigError(f'No config file specified')

        _config = ConfigParser()
        _config.read(config_file)

        _main_section = _config[MAIN]
        self.pg_bin_path = _main_section.get('pg_bin_path')
        self.pg_dump_flags = _main_section.get('pg_dump_flags', '')
        self.log_dir = _main_section.get('log_dir')
        self.teir1_days = _main_section.getint('teir1_days', 7)
        self.teir2_copies_interval = _main_section.getint('teir2_copies_interval', 7)
        self.tier2_store_days = _main_section.getint('tier2_store_days', 30)
        self.keep_logs_days = _main_section.getint('keep_logs_days', 30)

        self.smtp_server = None
        self.mail_from = None
        self.mail_to = None
        self.smtp_port = None

        if _config.has_section(SMTP):
            _smtp_section = _config[SMTP]
            self.smtp_server = _smtp_section.get('smtp server')
            self.mail_from = _smtp_section.get('mail from')
            self.mail_to = _smtp_section.get('mail to', fallback='root')
            self.smtp_port = _smtp_section.getint('smtp port', fallback=25)

        if not isdir(self.pg_bin_path):
            raise ConfigError(f'No valid directory with binnary files of PostgreSQL is set: {self.pg_bin_path}')

        self.tasks = list(CommonTaskDescription.parse_config(_config))
