# coding: utf-8

from .config import Config

from email.message import EmailMessage
from smtplib import SMTP, SMTPException
from typing import List, Optional
from socket import gethostname
from datetime import datetime
from os.path import basename, exists


class MailError(Exception):
    pass


class MailSender(object):
    def __init__(self,
                 server: Optional[str], port: Optional[int],
                 mail_from: Optional[str], mail_to: Optional[str]
                 ):

        self.server = server
        self.mail_from = mail_from
        self.mail_to = mail_to if mail_to is not None else 'root'
        self.port = port if port is not None else 25

    def sendmail(self, content: str, logs: List[str]) -> None:
        if self.server:
            msg = EmailMessage()
            msg['To'] = self.mail_to
            if self.mail_from:
                msg['From'] = self.mail_from

            msg['Subject'] = f'{gethostname()}: Posgres backup - {datetime.now()}'
            msg.set_content(content)
            msg.make_mixed()
            for f in logs:
                try:
                    if exists(f):
                        msg.add_attachment(open(f).read(), filename=basename(f))

                except IOError as e:
                    raise MailError(f'Error on open logfile "{f}": {e}')

            try:
                s = SMTP(host=self.server, port=self.port)
                s.helo(gethostname())
                s.sendmail(
                    from_addr=(self.mail_from if self.mail_from is not None else ''),
                    to_addrs=self.mail_to,
                    msg=msg.as_bytes()
                )
                s.quit()
            except SMTPException as e:
                raise MailError(f'Error on sending Email: {e}')

    @classmethod
    def from_config(cls, config: Config):
        return cls(
            server=config.smtp_server,
            mail_from=config.mail_from,
            mail_to=config.mail_to,
            port=config.smtp_port
        )