# coding: utf-8

import subprocess as sp
from os.path import join as p_join
from os import environ

from .store_controller import StoreController
from .log_controller import LogController
from .config import Config, DBTask
from .error import Error


class ExecutorError(Error):
    pass


def backup(task: DBTask, config: Config, log_controller: LogController):
    log = log_controller.get_logger(f'{task.name}.executor')
    log_t = log.get_timing()
    log(log_t(f'Начинаем копирование {task.name}'))

    stor = StoreController(task)
    backup_item = stor.new_item()

    log(log_t(f'Копируется "{task.db_name}" -> "{backup_item.get_path()}" '
              f'с параметрами pg_dump "{config.pg_dump_flags}"'))

    pg_dump = p_join(config.pg_bin_path, 'pg_dump.exe')
    pg_dump = (
        f'"{pg_dump}" {config.pg_dump_flags} -F c -v --clean '
        f'-U "{task.user_name}" -h "{task.host_name}" -p "{task.port}" '
        f'-f "{backup_item.get_path()}" {task.db_name}'
    )

    environ['PGPASSWORD'] = task.passwd

    pglog_name = log_controller.get_filename(task.name)
    pglog_fd = sp.DEVNULL if pglog_name is None else open(pglog_name, 'w')

    log(log_t(f'Пробуем запустить: {pg_dump}'))

    try:
        cmd = sp.Popen(pg_dump, stderr=sp.STDOUT, stdout=pglog_fd, shell=True)
        cmd.wait()

        log(log_t(f'Команда выполнена, статус возврата "{cmd.returncode}", журнал: "{pglog_name}"'))

        if cmd.returncode != 0:
            raise ExecutorError(f'Не нулевой код возврата, подробности в "{pglog_name}"')

    except:
        stor.remove(backup_item)
        raise

    else:
        stor.add_item(backup_item)

    cleaned = stor.clean(config.teir1_days, config.teir2_copies_interval, config.tier2_store_days)

    if cleaned:
        log(log_t('Очистка старых копий'))
        log(log_t('\n'.join(map(lambda x: f'- {x}', cleaned))))

    if stor.op_adv_status:
        log(log_t('Расширенное состояние хранимых копий второго класса'))
        log(log_t('\n'.join(map(lambda x: f'- {x}', stor.op_adv_status))))

    stor.save_index()
