# coding: utf-8
"""\
Модуль логирования

Метки в журнале о уровне сообщения:
  "`": Debug
  ".": Info
  "*": Warning
  "!": Error
  "#": Alert
"""

from time import time, ctime
from datetime import timedelta, datetime
from sys import exc_info
from os.path import join as p_join
from os import listdir, remove as file_remove, stat
from traceback import extract_tb, extract_stack
from typing import Optional

from .config import Config


class Timing(object):
    def __init__(self, name: Optional[str] = None):
        if name is None:
            self.prefix = ''
        else:
            self.prefix = f'{name} :: '
        self.tsAll = time()
        self.ts = self.tsAll

    def get_time(self):
        return time() - self.ts

    def reset(self):
        self.ts = time()
        self.tsAll = self.ts

    def __str__(self):
        ts = time()
        return self.prefix + '%s(%.4f)' % (timedelta(seconds=(ts - self.tsAll)), ts - self.ts)

    def __call__(self, msg):
        _buf = f'{self} | {msg}'
        self.ts = time()
        return _buf


class BaseLogger(object):
    def __init__(self, appname='main'):
        self.appname = appname

    @staticmethod
    def _write(itr_content):
        raise NotImplemented()

    def __call__(self, msg):
        self._write(map(
            lambda x: '%3s | %s :: %s' % ('.', self.appname, x),
            str(msg).splitlines()
        ))

    def err(self, msg):
        self._write(map(
            lambda x: '%3s | %s :: %s' % ('!', self.appname, x),
            str(msg).splitlines()
        ))

    def warn(self, msg):
        self._write(map(
            lambda x: '%3s | %s :: %s' % ('*', self.appname, x),
            str(msg).splitlines()
        ))

    def alert(self, msg):
        self._write(map(
            lambda x: '%3s | %s :: %s' % ('#', self.appname, x),
            str(msg).splitlines()
        ))

    def debug(self, msg):
        self._write(map(
            lambda x: '%3s | %s :: %s' % ('`', self.appname, x),
            str(msg).splitlines()
        ))

    @staticmethod
    def get_timing(name: Optional[str] = None):
        return Timing(name)

    def sublog(self, name):
        return self.__class__(f'{self.appname}/{name}')

    def excpt(self, msg, e_class=None, e_obj=None, e_tb=None, stack_skip=0):
        if e_class is None:
            e_class, e_obj, e_tb = exc_info()

        tb_data_tb = list(extract_tb(e_tb))[::-1]
        tb_data_stack = list(extract_stack())[::-1][(2 + stack_skip):]
        self.err(msg)
        self.err('--- EXCEPTION ---')
        self.err(' %s (%s)' % (e_class.__name__, e_obj))
        self.err('--- TRACEBACK ---')
        for _tbFile, _tbLine, _tbFunc, _tbText in tb_data_tb:
            self.err('File: %s, line %s in %s' % (_tbFile, _tbLine, _tbFunc))
            self.err('   %s' % _tbText)
        self.err('>>> Exception Handler <<<')
        for _tbFile, _tbLine, _tbFunc, _tbText in tb_data_stack:
            self.err('File: %s, line %s in %s' % (_tbFile, _tbLine, _tbFunc))
            self.err('   %s' % _tbText)
        self.err('--- END EXCEPTION ---')


class NullLogger(BaseLogger):
    @staticmethod
    def _write(itr_content):
        pass


class FileLogger(BaseLogger):
    def __init__(self, filename: str, appname: str = 'main'):
        super().__init__(appname)
        self.fd = open(filename, 'a')

    def _write(self, itr_content):
        cur_time = ctime()
        for i in itr_content:
            self.fd.write(f'{cur_time}{i}\n')

        self.fd.flush()

    def __del__(self):
        try:
            self.fd.close()
        except:
            pass


class LogController(object):
    def __init__(self, config: Config):
        self.log_dir = config.log_dir
        self.keep_logs_days = config.keep_logs_days

    @staticmethod
    def _get_timeprefix() -> str:
        return datetime.now().strftime('%Y-%m-%d_%H-%M-%S')

    def get_logger(self, name: str) -> BaseLogger:
        if self.log_dir is not None:
            return FileLogger(p_join(self.log_dir, f'{self._get_timeprefix()} - {name}.log'), appname=name)
        else:
            return NullLogger(appname=name)

    def get_filename(self, name: str) -> Optional[str]:
        if self.log_dir is not None:
            return str(p_join(self.log_dir, f'{self._get_timeprefix()} - {name}.log'))

    def clean(self):
        if self.log_dir is not None:
            now = time()
            for f in listdir(self.log_dir):
                f_path = p_join(self.log_dir, f)
                f_stat = stat(f_path)

                if divmod(now - f_stat.st_ctime, 86400)[0] > self.keep_logs_days:
                    file_remove(f_path)
