# coding: utf-8

import json
from datetime import datetime
from os.path import exists, join as p_join, basename
from os import remove as file_remove
from typing import Dict, List

from .error import Error
from .config import DBTask


class StoreError(Error):
    pass


class StoreControllerBase(object):
    def get_filename(self, filename: str) -> str:
        raise NotImplemented()


class IndexItem(object):
    def __init__(self, controller: StoreControllerBase, time: int, filename: str):
        self._controller = controller
        self.filename = filename
        self.time = time
        self._my_path = None

    def get_path(self):
        if self._my_path is None:
            self._my_path = self._controller.get_filename(self.filename)

        return self._my_path

    def get_datetime(self):
        return datetime.fromtimestamp(self.time)

    def is_exists(self) -> bool:
        return exists(self.get_path())

    def to_dict(self) -> dict:
        return dict((i, getattr(self, i)) for i in ('filename', 'time'))

    @classmethod
    def from_dict(cls, controller: StoreControllerBase, d: Dict):
        return cls(
            controller=controller,
            **d
        )

    @classmethod
    def new(cls, controller: StoreControllerBase, short_filename: str):
        time = datetime.now()
        time_prefix = time.strftime('%Y-%m-%d_%H-%M-%S')
        filename = f'{time_prefix} - {short_filename}'

        return cls(
            controller=controller,
            time=int(time.timestamp()),
            filename=filename
        )


class StoreController(StoreControllerBase):
    def __init__(self, task: DBTask):
        self.task = task
        self.index_name = self.get_filename(f'{task.name}.index')
        self.idx: Dict[int, IndexItem] = {}

        if exists(self.index_name):
            self.load_index()

    def get_filename(self, filename: str) -> str:
        return str(p_join(self.task.dst_dir, filename))

    def load_index(self) -> None:
        with open(self.index_name, 'r') as IN:
            result = {}
            for itm in json.load(IN):
                itm = IndexItem.from_dict(self, itm)
                if itm.is_exists():
                    result[itm.time] = itm

        self.idx = result

    def save_index(self) -> None:
        with open(self.index_name, 'w') as OUT:
            json.dump(list(map(lambda x: x.to_dict(), self.idx.values())), OUT)

    def new_item(self) -> IndexItem:
        return IndexItem.new(self, f'{self.task.name}.backup')

    def add_item(self, item: IndexItem) -> None:
        item_path = item.get_path()
        if not item.is_exists():
            raise StoreError(f'Storing to index file not found: {item.get_path()}')

        if item.time in self.idx and self.idx[item.time].filename != item.filename:
            if self.idx[item.time].is_exists():
                file_remove(self.idx[item.time].get_path())

        self.idx[item.time] = item

    def remove(self, item: IndexItem) -> None:
        if item.time in self.idx:
            del self.idx[item.time]

        if item.is_exists():
            file_remove(item.get_path())

    def __iter__(self):
        for i in sorted(self.idx):
            yield self.idx[i]

    def clean(self, tier1_days: int, tier2_copies_interval: int, tier2_store_days: int) -> List[str]:
        to_remove = []
        tier2_idx = {}
        now = datetime.now()

        for item in self:
            if not item.is_exists():
                to_remove.append(item)

            else:
                storing_days = (now - item.get_datetime()).days

                if not storing_days <= tier1_days:
                    if storing_days > tier2_store_days:
                        to_remove.append(item)

                    else:
                        # Магия: Делим старые резервные копии на эпохи. Эпоха - целая часть от деления количества
                        #        дней, которое лежит резервная копия, на интервал, за который нам нужно хранить
                        #        хотя бы одну копию. В одной эпохе старший файл вытесняет младший. Из вычислений
                        #        убираем период tier1

                        storing_days -= tier1_days

                        _epoch = divmod(storing_days, tier2_copies_interval)[0]
                        if _epoch in tier2_idx:
                            to_remove.append(tier2_idx[_epoch])

                        tier2_idx[_epoch] = item

        result = []
        for item in to_remove:
            file_remove(item.get_path())
            result.append(item.filename)

        return result
