# coding: utf-8

import json
from datetime import datetime
from os.path import exists, join as p_join
from os import remove as file_remove
from typing import Dict, List
from dataclasses import dataclass, asdict
from os import stat as f_stat

from .error import Error
from .config import DBTask


class StoreError(Error):
    pass


class StoreControllerBase(object):
    def get_filename(self, filename: str) -> str:
        raise NotImplemented()


class IndexItem(object):
    def __init__(self, controller: StoreControllerBase, time: int, filename: str, epoch: int = 0):
        self._controller = controller
        self.filename = filename
        self.time = time
        self.epoch = epoch
        self._my_path = None

    def get_path(self):
        if self._my_path is None:
            self._my_path = self._controller.get_filename(self.filename)

        return self._my_path

    def set_epoch(self, num_days: int):
        """\
        Устанавливает значение эпохи для элемиента в соответствии с количеством дней в эпохе
        """
        self.epoch = divmod(self.time, num_days * 86400)[0]  # 86400 - секунд в сутках

    def __str__(self) -> str:
        return self.filename

    def full_desc(self) -> str:
        return f'{self.filename} [epoch="{self.epoch}" created={self.get_datetime()}]'

    def get_datetime(self):
        return datetime.fromtimestamp(self.time)

    def is_exists(self) -> bool:
        return exists(self.get_path())

    def to_dict(self) -> dict:
        return dict((i, getattr(self, i)) for i in ('filename', 'time', 'epoch'))

    @classmethod
    def from_dict(cls, controller: StoreControllerBase, d: Dict):
        return cls(
            controller=controller,
            **d
        )

    @classmethod
    def new(cls, controller: StoreControllerBase, short_filename: str):
        time = datetime.now()
        time_prefix = time.strftime('%Y-%m-%d_%H-%M-%S')
        filename = f'{time_prefix} - {short_filename}'

        return cls(
            controller=controller,
            time=int(time.timestamp()),
            filename=filename
        )


@dataclass()
class TaskStatus(object):
    status: bool = False
    size: int = 0
    time: int = 0


class StoreController(StoreControllerBase):
    def __init__(self, task: DBTask):
        self.task = task
        self.index_name = self.get_filename(f'{task.name}.index')
        self.status_name = self.get_filename(f'{task.name}.status')
        self.idx: Dict[int, IndexItem] = {}
        self.op_adv_status = []

        if exists(self.index_name):
            self.load_index()

    def get_filename(self, filename: str) -> str:
        return str(p_join(self.task.dst_dir, filename))

    def load_index(self) -> None:
        with open(self.index_name, 'r') as IN:
            result = {}
            for itm in json.load(IN):
                itm = IndexItem.from_dict(self, itm)
                if itm.is_exists():
                    result[itm.time] = itm

        self.idx = result

    def set_epoch(self, num_days: int):
        for i in self.idx.values():
            i.set_epoch(num_days)

    def save_index(self) -> None:
        with open(self.index_name, 'w') as OUT:
            json.dump(list(map(lambda x: x.to_dict(), self.idx.values())), OUT)

    def add_op_status(self, msg: str) -> None:
        self.op_adv_status.append(msg)

    def new_item(self) -> IndexItem:
        return IndexItem.new(self, f'{self.task.name}.backup')

    def add_item(self, item: IndexItem) -> None:
        if not item.is_exists():
            raise StoreError(f'Storing to index file not found: {item.get_path()}')

        if item.time in self.idx and self.idx[item.time].filename != item.filename:
            if self.idx[item.time].is_exists():
                file_remove(self.idx[item.time].get_path())

        self.idx[item.time] = item

    def remove(self, item: IndexItem) -> None:
        if item.time in self.idx:
            del self.idx[item.time]

        if item.is_exists():
            file_remove(item.get_path())

    def status(self, last_status: bool):
        res = TaskStatus(status=last_status)

        for i in self.idx.values():
            if res.time < i.time:
                res.time = i.time

            res.size += f_stat(i.get_path()).st_size

        with open(self.status_name, 'w') as OUT:
            json.dump(asdict(res), OUT)

    def __iter__(self):
        for i in sorted(self.idx, reverse=True):
            yield self.idx[i]

    def clean(self, tier1_days: int, tier2_store_days: int, epoch_days: int) -> List[str]:
        to_remove = []
        tier2_idx = {}
        now = datetime.now()

        self.set_epoch(epoch_days)

        for item in self:
            if not item.is_exists():
                to_remove.append(item)

            else:
                storing_days = (now - item.get_datetime()).days

                if not storing_days <= tier1_days:
                    if storing_days > tier2_store_days:
                        to_remove.append(item)

                    else:
                        tier2_storing_days = storing_days - tier1_days

                        self.add_op_status(f'"{item.full_desc()}": storing_days={storing_days}'
                                           f' tier2_storing_days={tier2_storing_days}')

                        if item.epoch in tier2_idx:
                            to_remove.append(tier2_idx[item.epoch])

                        tier2_idx[item.epoch] = item

        result = []
        for item in to_remove:
            file_remove(item.get_path())
            result.append(item.filename)

        return result
